<?php
/**
 * ISLMS - 學生批次匯入
 */
include_once 'src/function.php';
check_staff();

$result = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['import_file'])) {
    if (!verify_csrf($_POST['csrfToken'] ?? '')) {
        set_flash('error', 'CSRF 驗證失敗');
        redirect('students_import.php');
    }
    
    $file = $_FILES['import_file'];
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        set_flash('error', '檔案上傳失敗');
        redirect('students_import.php');
    }
    
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, ['csv', 'xlsx', 'xls'])) {
        set_flash('error', '僅支援 CSV、XLSX、XLS 格式');
        redirect('students_import.php');
    }
    
    $result = ['success' => 0, 'fail' => 0, 'errors' => []];
    
    // 處理 CSV 檔案
    if ($ext === 'csv') {
        $handle = fopen($file['tmp_name'], 'r');
        
        // 偵測編碼
        $content = file_get_contents($file['tmp_name']);
        $encoding = mb_detect_encoding($content, ['UTF-8', 'Big5', 'GB2312', 'ASCII']);
        if ($encoding && $encoding !== 'UTF-8') {
            $content = mb_convert_encoding($content, 'UTF-8', $encoding);
            file_put_contents($file['tmp_name'], $content);
        }
        
        $handle = fopen($file['tmp_name'], 'r');
        $header = fgetcsv($handle); // 讀取標題行
        $lineNum = 1;
        
        while (($row = fgetcsv($handle)) !== false) {
            $lineNum++;
            
            if (count($row) < 3) {
                $result['errors'][] = "第 $lineNum 行：資料欄位不足";
                $result['fail']++;
                continue;
            }
            
            $studentNo = trim($row[0] ?? '');
            $name = trim($row[1] ?? '');
            $email = trim($row[2] ?? '');
            $chineseName = trim($row[3] ?? '');
            $nationality = trim($row[4] ?? '');
            $levelCode = trim($row[5] ?? '');
            $studentType = trim($row[6] ?? 'exchange');
            $phone = trim($row[7] ?? '');
            
            if (empty($studentNo) || empty($name)) {
                $result['errors'][] = "第 $lineNum 行：學號或姓名為空";
                $result['fail']++;
                continue;
            }
            
            // 檢查學號是否重複
            $stmt = $conn->prepare("SELECT student_id FROM students WHERE student_no = ?");
            $stmt->execute([$studentNo]);
            if ($stmt->fetch()) {
                $result['errors'][] = "第 $lineNum 行：學號 $studentNo 已存在";
                $result['fail']++;
                continue;
            }
            
            try {
                $stmt = $conn->prepare("INSERT INTO students (student_no, name, chinese_name, email, nationality, level_code, student_type, phone, status, created_by) VALUES (?,?,?,?,?,?,?,?,1,?)");
                $stmt->execute([$studentNo, $name, $chineseName, $email, $nationality, $levelCode, $studentType ?: 'exchange', $phone, $_SESSION['user_account']]);
                $result['success']++;
            } catch (Exception $e) {
                $result['errors'][] = "第 $lineNum 行：匯入失敗 - " . $e->getMessage();
                $result['fail']++;
            }
        }
        
        fclose($handle);
    } else {
        // Excel 檔案處理 (需要 PhpSpreadsheet)
        set_flash('error', '請使用 CSV 格式匯入，或安裝 PhpSpreadsheet 套件以支援 Excel');
        redirect('students_import.php');
    }
    
    write_log('import', 'students', null, "匯入學生：成功 {$result['success']} 筆，失敗 {$result['fail']} 筆");
}

$levels = get_levels();
?>
<!DOCTYPE html>
<html lang="zh-TW">
<head>
  <meta charset="utf-8">
  <meta content="width=device-width, initial-scale=1.0" name="viewport">
  <title>學生批次匯入 - <?= APP_NAME ?></title>
  <?php include_once "temp_head.php"; ?>
</head>
<body>
  <?php include_once "temp_header.php"; ?>
  <?php include_once "temp_sidebar.php"; ?>

  <main id="main" class="main">
    <div class="pagetitle">
      <h1>學生批次匯入</h1>
      <nav>
        <ol class="breadcrumb">
          <li class="breadcrumb-item"><a href="index.php">Home</a></li>
          <li class="breadcrumb-item"><a href="students_list.php">學生管理</a></li>
          <li class="breadcrumb-item active">批次匯入</li>
        </ol>
      </nav>
    </div>

    <div class="container-fluid">
      <?php show_flash(); ?>
      
      <?php if ($result): ?>
      <div class="alert alert-<?= $result['fail'] > 0 ? 'warning' : 'success' ?>">
        <h5><i class="bi bi-<?= $result['fail'] > 0 ? 'exclamation-triangle' : 'check-circle' ?> me-2"></i>匯入結果</h5>
        <p>成功：<strong><?= $result['success'] ?></strong> 筆，失敗：<strong><?= $result['fail'] ?></strong> 筆</p>
        <?php if (!empty($result['errors'])): ?>
        <hr>
        <details>
          <summary>查看錯誤詳情</summary>
          <ul class="mt-2 mb-0">
            <?php foreach ($result['errors'] as $err): ?>
            <li><?= e($err) ?></li>
            <?php endforeach; ?>
          </ul>
        </details>
        <?php endif; ?>
      </div>
      <?php endif; ?>
      
      <div class="row">
        <div class="col-md-6">
          <div class="card">
            <div class="card-header"><i class="bi bi-upload me-2"></i>上傳檔案</div>
            <div class="card-body mt-3">
              <form method="post" enctype="multipart/form-data">
                <?= csrf_field() ?>
                <div class="mb-3">
                  <label class="form-label">選擇檔案 <span class="text-danger">*</span></label>
                  <input type="file" name="import_file" class="form-control" accept=".csv,.xlsx,.xls" required>
                  <div class="form-text">支援格式：CSV, XLSX, XLS</div>
                </div>
                <button type="submit" class="btn btn-primary">
                  <i class="bi bi-upload me-1"></i>開始匯入
                </button>
              </form>
            </div>
          </div>
        </div>
        
        <div class="col-md-6">
          <div class="card">
            <div class="card-header"><i class="bi bi-info-circle me-2"></i>匯入格式說明</div>
            <div class="card-body">
              <p>請依照以下欄位順序準備 CSV 檔案（第一行為標題）：</p>
              <table class="table table-sm table-bordered">
                <thead>
                  <tr class="table-light">
                    <th>欄位</th>
                    <th>必填</th>
                    <th>說明</th>
                  </tr>
                </thead>
                <tbody>
                  <tr><td>學號</td><td><span class="text-danger">*</span></td><td>唯一識別碼</td></tr>
                  <tr><td>姓名</td><td><span class="text-danger">*</span></td><td>英文姓名</td></tr>
                  <tr><td>Email</td><td></td><td>電子郵件</td></tr>
                  <tr><td>中文姓名</td><td></td><td></td></tr>
                  <tr><td>國籍</td><td></td><td></td></tr>
                  <tr><td>程度</td><td></td><td>BEG, BAS, ELE1, ELE2, INT1, INT2, INT3, ADV1</td></tr>
                  <tr><td>類型</td><td></td><td>exchange 或 degree</td></tr>
                  <tr><td>電話</td><td></td><td></td></tr>
                </tbody>
              </table>
              
              <a href="#" class="btn btn-outline-success btn-sm" onclick="downloadTemplate()">
                <i class="bi bi-download me-1"></i>下載範本
              </a>
            </div>
          </div>
        </div>
      </div>
    </div>
  </main>

  <?php include_once "temp_footer.php"; ?>

  <script>
    showSidebar("students", "students_import");
    
    function downloadTemplate() {
      const header = '學號,姓名,Email,中文姓名,國籍,程度,類型,電話\n';
      const sample = 'S12345678,John Smith,john@example.com,約翰史密斯,USA,BEG,exchange,0912345678\n';
      
      const blob = new Blob(['\ufeff' + header + sample], { type: 'text/csv;charset=utf-8;' });
      const link = document.createElement('a');
      link.href = URL.createObjectURL(blob);
      link.download = 'students_template.csv';
      link.click();
    }
  </script>
</body>
</html>
