<?php
/**
 * ISLMS - 共用函數庫
 * International Student Learning Management System
 */
include_once __DIR__ . '/globarvar.php';

// ==================== 亂數與Token ====================

function crypto_rand_secure($min, $max)
{
    $range = $max - $min;
    if ($range < 1) return $min;
    $log = ceil(log($range, 2));
    $bytes = (int) ($log / 8) + 1;
    $bits = (int) $log + 1;
    $filter = (int) (1 << $bits) - 1;
    do {
        $rnd = hexdec(bin2hex(openssl_random_pseudo_bytes($bytes)));
        $rnd = $rnd & $filter;
    } while ($rnd >= $range);
    return $min + $rnd;
}

function getToken($length)
{
    $token = "";
    $codeAlphabet = "ABCDEFGHJKLMNPQRSTUVWXYZ";
    $codeAlphabet .= "abcdefghijkmnopqrstuvwxyz";
    $codeAlphabet .= "123456789";
    $max = strlen($codeAlphabet) - 1;
    for ($i = 0; $i < $length; $i++) {
        $token .= $codeAlphabet[crypto_rand_secure(0, $max)];
    }
    return $token;
}

// ==================== 通用資料操作 ====================

function insert_data1($tablename, $fieldarray)
{
    global $conn;
    $n_rid = 0;
    try {
        $columns = implode(", ", array_keys($fieldarray));
        $placeholders = implode(", ", array_fill(0, count($fieldarray), "?"));
        
        $sql = "INSERT INTO $tablename ($columns) VALUES ($placeholders) ON DUPLICATE KEY UPDATE ";
        
        $updateAssignments = [];
        foreach ($fieldarray as $key => $value) {
            $updateAssignments[] = "$key = VALUES($key)";
        }
        $sql .= implode(", ", $updateAssignments);
        
        $stmt = $conn->prepare($sql);
        $stmt->execute(array_values($fieldarray));
        $n_rid = $conn->lastInsertId();
    } catch (PDOException $e) {
        // Error handling
    }
    return $n_rid;
}

function update_data($tablename, $fieldarray, $where_clause = '')
{
    global $conn;
    try {
        $update_parts = [];
        foreach ($fieldarray as $field => $value) {
            $update_parts[] = "$field = :$field";
        }
        $update_str = implode(", ", $update_parts);
        $sql = "UPDATE {$tablename} SET {$update_str}";
        if (!empty($where_clause)) {
            $sql .= " WHERE {$where_clause}";
        }
        
        $stmt = $conn->prepare($sql);
        foreach ($fieldarray as $field => $value) {
            $stmt->bindValue(":$field", $value);
        }
        $stmt->execute($fieldarray);
    } catch (PDOException $e) {
        return false;
    }
    return true;
}

function checkData($data, $type)
{
    if ($type === 'string') {
        if (empty($data) || !preg_match("/^[\p{L}\p{N} _\/\.,;:!?\"()\-。，、！？\r\n&'#\' ]+$/u", $data)) {
            return null;
        } else {
            $data = str_replace("&#039;", "'", $data);
            $data = str_replace("&amp;", "&", $data);
            $data = str_replace("&quot;", '"', $data);
            return $data;
        }
    } else if ($type === 'number') {
        if (empty($data) || !preg_match("/^-?[0-9]+$/", $data)) {
            return null;
        }
    } else if ($type === 'email') {
        if (empty($data) || !filter_var($data, FILTER_VALIDATE_EMAIL)) {
            return null;
        }
    } else if ($type === 'date') {
        if (empty($data) || !preg_match("/^\d{4}-\d{2}-\d{2}$/", $data)) {
            return null;
        }
    }
    return $data;
}

// ==================== 登入驗證 ====================

/**
 * 檢查登入狀態
 */
function check_login()
{
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_account'])) {
        header("Location: login.php");
        exit;
    }
    // 延長 session 的存活時間 (2 小時)
    $session_lifetime = 7200;
    setcookie(session_name(), session_id(), time() + $session_lifetime, "/");
    return true;
}

/**
 * 檢查管理員權限
 */
function check_admin()
{
    if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] != 'admin') {
        header("Location: login.php");
        exit;
    }
    return true;
}

/**
 * 檢查教務人員權限 (admin 或 staff)
 */
function check_staff()
{
    check_login();
    if (!in_array($_SESSION['user_role'], ['admin', 'staff'])) {
        header("Location: index.php");
        exit;
    }
    return true;
}

/**
 * 檢查是否有角色權限
 */
function has_role($roles)
{
    if (!is_array($roles)) {
        $roles = [$roles];
    }
    return in_array($_SESSION['user_role'] ?? '', $roles);
}

/**
 * 登入函數
 */
function login($account, $password)
{
    global $conn;
    $today = date("Y-m-d");
    $sql = "SELECT * FROM users WHERE account = ? LIMIT 1";
    try {
        $stmt = $conn->prepare($sql);
        $stmt->execute([$account]);
        $result = $stmt->fetch();
        
        if (is_array($result)) {
            if (isset($result['status']) && $result['status'] == 0) {
                return '帳號已停用!';
            } else if (isset($result['password_hash']) && password_verify($password, $result['password_hash'])) {
                $_SESSION['user_id'] = $result['user_id'];
                $_SESSION['user_account'] = $result['account'];
                $_SESSION['user_name'] = $result['name'];
                $_SESSION['user_role'] = $result['role'] ?? 'staff';
                
                // 更新最後登入時間
                $stmt = $conn->prepare("UPDATE users SET last_login_at = NOW() WHERE user_id = ?");
                $stmt->execute([$result['user_id']]);
                
                writeSystemLog('LOGIN', '使用者登入');
                return '登入成功!';
            } else {
                return '帳號或密碼錯誤!';
            }
        } else {
            return '帳號或密碼錯誤!';
        }
    } catch (PDOException $e) {
        return '發生錯誤!';
    }
}

/**
 * HTML 跳脫
 */
function e($str) {
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * 產生 CSRF hidden field
 */
function csrf_field() {
    return '<input type="hidden" name="csrfToken" value="' . ($_SESSION['csrfToken'] ?? '') . '">';
}

/**
 * 驗證 CSRF Token
 */
function verify_csrf($token) {
    return isset($_SESSION['csrfToken']) && hash_equals($_SESSION['csrfToken'], $token);
}

/**
 * 設定 Flash 訊息
 */
function set_flash($type, $message) {
    $_SESSION['flash'] = ['type' => $type, 'message' => $message];
}

/**
 * 取得並清除 Flash 訊息
 */
function get_flash() {
    $flash = $_SESSION['flash'] ?? null;
    unset($_SESSION['flash']);
    return $flash;
}

/**
 * 顯示 Flash 訊息 HTML
 */
function show_flash() {
    $flash = get_flash();
    if ($flash) {
        $type = $flash['type'] === 'error' ? 'danger' : $flash['type'];
        echo '<div class="alert alert-' . e($type) . ' alert-dismissible fade show" role="alert">';
        echo e($flash['message']);
        echo '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>';
        echo '</div>';
    }
}

/**
 * 日期格式化
 */
function format_date($date, $format = 'Y-m-d') {
    if (empty($date)) return '';
    return date($format, strtotime($date));
}

/**
 * 日期時間格式化
 */
function format_datetime($datetime, $format = 'Y-m-d H:i') {
    if (empty($datetime)) return '';
    return date($format, strtotime($datetime));
}

/**
 * 取得程度名稱
 */
function get_level_name($code) {
    $levels = [
        'BEG' => '入門華語 Beginner',
        'BAS' => '基礎華語 Basic',
        'ELE1' => '初級華語一 Elementary I',
        'ELE2' => '初級華語二 Elementary II',
        'INT1' => '中級華語一 Intermediate I',
        'INT2' => '中級華語二 Intermediate II',
        'INT3' => '中級華語三 Intermediate III',
        'ADV1' => '進階華語一 Advanced I'
    ];
    return $levels[$code] ?? $code;
}

/**
 * 取得出席狀態
 */
function get_attendance_status($code) {
    $statuses = [
        'P' => ['name' => '出席', 'class' => 'success'],
        'L' => ['name' => '遲到', 'class' => 'warning'],
        'A' => ['name' => '缺席', 'class' => 'danger'],
        'E' => ['name' => '請假', 'class' => 'info'],
        'O' => ['name' => '公假', 'class' => 'primary'],
        'S' => ['name' => '病假', 'class' => 'secondary']
    ];
    return $statuses[$code] ?? ['name' => $code, 'class' => 'secondary'];
}

/**
 * 取得角色名稱
 */
function get_role_name($role) {
    $roles = [
        'admin' => '系統管理員',
        'staff' => '教務承辦',
        'teacher' => '授課教師'
    ];
    return $roles[$role] ?? $role;
}

/**
 * 取得開班狀態
 */
function get_offering_status($status) {
    $statuses = [
        'draft' => ['name' => '草稿', 'class' => 'secondary'],
        'active' => ['name' => '進行中', 'class' => 'success'],
        'closed' => ['name' => '已結束', 'class' => 'dark']
    ];
    return $statuses[$status] ?? ['name' => $status, 'class' => 'secondary'];
}

/**
 * 取得選課狀態
 */
function get_enrollment_status($status) {
    $statuses = [
        'active' => ['name' => '在學中', 'class' => 'success'],
        'dropped' => ['name' => '已退選', 'class' => 'warning'],
        'withdrawn' => ['name' => '已退學', 'class' => 'danger'],
        'completed' => ['name' => '已結業', 'class' => 'info']
    ];
    return $statuses[$status] ?? ['name' => $status, 'class' => 'secondary'];
}

/**
 * 取得等第
 */
function get_grade_letter($score) {
    if ($score === null) return '';
    if ($score >= 80) return 'A';
    if ($score >= 70) return 'B';
    if ($score >= 60) return 'C';
    return 'D';
}

/**
 * 取得等第 Badge Class
 */
function get_grade_class($grade) {
    $classes = ['A' => 'success', 'B' => 'primary', 'C' => 'warning', 'D' => 'danger'];
    return $classes[$grade] ?? 'secondary';
}

/**
 * 寫入系統日誌
 */
function write_log($action, $tableName = null, $recordId = null, $description = null) {
    global $conn, $mUserId, $client_ip;
    
    $sql = "INSERT INTO audit_logs (user_id, action, table_name, record_id, description, ip_address, created_at) 
            VALUES (:user_id, :action, :table_name, :record_id, :description, :ip, NOW())";
    $stmt = $conn->prepare($sql);
    $stmt->execute([
        ':user_id' => $mUserId ?: null,
        ':action' => $action,
        ':table_name' => $tableName,
        ':record_id' => $recordId,
        ':description' => $description,
        ':ip' => $client_ip
    ]);
}

/**
 * 記錄系統日誌 (相容格式)
 */
function writeSystemLog(string $action_type, string $action_detail): bool
{
    global $conn;
    $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    
    $sql = "INSERT INTO audit_logs (user_id, action, description, ip_address, created_at) 
            VALUES (:user_id, :action, :description, :ip, NOW())";
    
    try {
        $stmt = $conn->prepare($sql);
        $user_id = $_SESSION['user_id'] ?? null;
        $stmt->bindValue(':user_id', $user_id, $user_id !== null ? PDO::PARAM_INT : PDO::PARAM_NULL);
        $stmt->bindValue(':action', $action_type, PDO::PARAM_STR);
        $stmt->bindValue(':description', $action_detail, PDO::PARAM_STR);
        $stmt->bindValue(':ip', $ip, PDO::PARAM_STR);
        return $stmt->execute();
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * 取得客戶端 IP
 */
function getClientIP()
{
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        $ip = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        $ip = $_SERVER['REMOTE_ADDR'];
    }
    return $ip;
}

/**
 * 傳送 JSON 回應
 */
function send_response($success, $message, $data = null)
{
    $response = [
        'success' => $success,
        'message' => $message
    ];
    
    if ($data !== null) {
        $response['data'] = $data;
    }
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * 回傳 JSON 成功
 */
function json_success($message = 'success', $data = null) {
    header('Content-Type: application/json; charset=utf-8');
    send_response(true, $message, $data);
}

/**
 * 回傳 JSON 錯誤
 */
function json_error($message = 'error', $data = null) {
    header('Content-Type: application/json; charset=utf-8');
    send_response(false, $message, $data);
}

/**
 * 密碼雜湊
 */
function hash_password($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * 驗證密碼
 */
function verify_password($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * 產生隨機密碼
 */
function generate_password($length = 8) {
    $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    return substr(str_shuffle($chars), 0, $length);
}

/**
 * 回傳 JSON 成功
 */
function json_success($message = 'success', $data = null) {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['success' => true, 'message' => $message, 'data' => $data], JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * 回傳 JSON 錯誤
 */
function json_error($message = 'error', $data = null) {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['success' => false, 'message' => $message, 'data' => $data], JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * 取得分頁 SQL
 */
function get_pagination($totalItems, $itemsPerPage = 10, $page = 1) {
    $page = max(1, intval($page));
    $totalPages = ceil($totalItems / $itemsPerPage);
    $offset = ($page - 1) * $itemsPerPage;
    
    return [
        'page' => $page,
        'totalPages' => $totalPages,
        'totalItems' => $totalItems,
        'itemsPerPage' => $itemsPerPage,
        'offset' => $offset
    ];
}

/**
 * 顯示分頁導航
 */
function render_pagination($pagination, $baseUrl = '?') {
    $page = $pagination['page'];
    $totalPages = $pagination['totalPages'];
    $totalItems = $pagination['totalItems'];
    $itemsPerPage = $pagination['itemsPerPage'];
    
    if ($totalPages <= 1) return '';
    
    $displayPages = 5;
    $startPage = max(1, $page - floor($displayPages / 2));
    $endPage = min($totalPages, $startPage + $displayPages - 1);
    
    if ($endPage - $startPage + 1 < $displayPages) {
        $startPage = max(1, $endPage - $displayPages + 1);
    }
    
    $html = '<div class="d-flex justify-content-center align-items-center mt-4">';
    $html .= '<span class="me-3">共 ' . $totalItems . ' 筆，每頁 ' . $itemsPerPage . ' 筆</span>';
    $html .= '<nav><ul class="pagination mb-0">';
    
    // 上一頁
    if ($page > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=' . ($page - 1) . '">&laquo;</a></li>';
    }
    
    // 第一頁
    if ($startPage > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=1">1</a></li>';
        if ($startPage > 2) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
    }
    
    // 頁碼
    for ($i = $startPage; $i <= $endPage; $i++) {
        $active = $i == $page ? ' active' : '';
        $html .= '<li class="page-item' . $active . '"><a class="page-link" href="' . $baseUrl . 'page=' . $i . '">' . $i . '</a></li>';
    }
    
    // 最後一頁
    if ($endPage < $totalPages) {
        if ($endPage < $totalPages - 1) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=' . $totalPages . '">' . $totalPages . '</a></li>';
    }
    
    // 下一頁
    if ($page < $totalPages) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . 'page=' . ($page + 1) . '">&raquo;</a></li>';
    }
    
    $html .= '</ul></nav></div>';
    
    return $html;
}

/**
 * 取得系統設定值
 */
function get_setting($key, $default = null) {
    global $conn;
    $stmt = $conn->prepare("SELECT setting_value, setting_type FROM system_settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $row = $stmt->fetch();
    
    if (!$row) return $default;
    
    $value = $row['setting_value'];
    switch ($row['setting_type']) {
        case 'int': return intval($value);
        case 'float': return floatval($value);
        case 'boolean': return $value === '1' || $value === 'true';
        case 'json': return json_decode($value, true);
        default: return $value;
    }
}

/**
 * 取得系統參數值 (相容格式)
 */
function getSystemSetting($setting_key) {
    global $conn;
    
    $sql = "SELECT setting_value FROM system_settings WHERE setting_key = :setting_key LIMIT 1";
    $stmt = $conn->prepare($sql);
    $stmt->execute([':setting_key' => $setting_key]);
    
    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        return $row['setting_value'];
    } else {
        return null;
    }
}

/**
 * 設定系統設定值
 */
function set_setting($key, $value, $type = 'string') {
    global $conn;
    
    if ($type === 'json') {
        $value = json_encode($value, JSON_UNESCAPED_UNICODE);
    } elseif ($type === 'boolean') {
        $value = $value ? '1' : '0';
    }
    
    $stmt = $conn->prepare("INSERT INTO system_settings (setting_key, setting_value, setting_type, updated_at) 
                            VALUES (?, ?, ?, NOW()) 
                            ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), updated_at = NOW()");
    $stmt->execute([$key, $value, $type]);
}

/**
 * 取得教師列表
 */
function get_teachers() {
    global $conn;
    $stmt = $conn->query("SELECT user_id, name, name_en FROM users WHERE role = 'teacher' AND status = 1 ORDER BY name");
    return $stmt->fetchAll();
}

/**
 * 取得程度列表
 */
function get_levels() {
    global $conn;
    $stmt = $conn->query("SELECT * FROM levels WHERE status = 1 ORDER BY sort_order");
    return $stmt->fetchAll();
}

/**
 * 取得課程列表
 */
function get_courses($levelCode = null) {
    global $conn;
    $sql = "SELECT * FROM courses WHERE status = 1";
    if ($levelCode) {
        $sql .= " AND level_code = ?";
        $stmt = $conn->prepare($sql);
        $stmt->execute([$levelCode]);
    } else {
        $stmt = $conn->query($sql . " ORDER BY level_code, course_code");
    }
    return $stmt->fetchAll();
}

/**
 * 安全重導向
 */
function redirect($url) {
    header('Location: ' . $url);
    exit;
}
