<?php
/**
 * ISLMS - 班級詳細/名單管理
 */
include_once 'src/function.php';
check_login();

$sectionId = intval($_GET['id'] ?? 0);
if (!$sectionId) {
    set_flash('error', '參數錯誤');
    redirect('sections_list.php');
}

// 取得班級資訊
$stmt = $conn->prepare("SELECT s.*, o.semester, o.offering_id, c.course_name, c.level_code, u.name as teacher_name 
                        FROM sections s
                        JOIN offerings o ON s.offering_id = o.offering_id
                        JOIN courses c ON o.course_id = c.course_id
                        LEFT JOIN users u ON COALESCE(s.teacher_id, o.teacher_id) = u.user_id
                        WHERE s.section_id = ?");
$stmt->execute([$sectionId]);
$section = $stmt->fetch();

if (!$section) {
    set_flash('error', '班級不存在');
    redirect('sections_list.php');
}

// 教師權限檢查
if ($_SESSION['user_role'] === 'teacher') {
    $stmt = $conn->prepare("SELECT 1 FROM sections s JOIN offerings o ON s.offering_id = o.offering_id WHERE s.section_id = ? AND (s.teacher_id = ? OR o.teacher_id = ?)");
    $stmt->execute([$sectionId, $_SESSION['user_id'], $_SESSION['user_id']]);
    if (!$stmt->fetch()) {
        set_flash('error', '您沒有權限查看此班級');
        redirect('sections_list.php');
    }
}

// 取得班級學生名單
$stmt = $conn->prepare("SELECT s.*, e.enrollment_id, e.enrollment_status, e.final_score, e.final_grade, e.attendance_rate, e.enrolled_at
                        FROM students s
                        JOIN enrollments e ON s.student_id = e.student_id
                        WHERE e.section_id = ?
                        ORDER BY e.enrollment_status, s.student_no");
$stmt->execute([$sectionId]);
$students = $stmt->fetchAll();

// 取得其他班級（用於調班）
$stmt = $conn->prepare("SELECT * FROM sections WHERE offering_id = ? AND section_id != ? ORDER BY section_name");
$stmt->execute([$section['offering_id'], $sectionId]);
$otherSections = $stmt->fetchAll();

$canEdit = has_role(['admin', 'staff']);
?>
<!DOCTYPE html>
<html lang="zh-TW">
<head>
  <meta charset="utf-8">
  <meta content="width=device-width, initial-scale=1.0" name="viewport">
  <title><?= e($section['section_name']) ?> - <?= APP_NAME ?></title>
  <?php include_once "temp_head.php"; ?>
</head>
<body>
  <?php include_once "temp_header.php"; ?>
  <?php include_once "temp_sidebar.php"; ?>

  <main id="main" class="main">
    <div class="pagetitle">
      <h1><?= e($section['course_name']) ?> - <?= e($section['section_name']) ?></h1>
      <nav>
        <ol class="breadcrumb">
          <li class="breadcrumb-item"><a href="index.php">Home</a></li>
          <li class="breadcrumb-item"><a href="sections_list.php">班級管理</a></li>
          <li class="breadcrumb-item active"><?= e($section['section_name']) ?></li>
        </ol>
      </nav>
    </div>

    <div class="container-fluid">
      <?php show_flash(); ?>
      
      <!-- 班級資訊 -->
      <div class="row mb-4">
        <div class="col-md-8">
          <div class="card h-100">
            <div class="card-header"><i class="bi bi-info-circle me-2"></i>班級資訊</div>
            <div class="card-body">
              <div class="row">
                <div class="col-md-6">
                  <p><strong>學期：</strong><?= e($section['semester']) ?></p>
                  <p><strong>程度：</strong><span class="badge bg-info"><?= e($section['level_code']) ?></span></p>
                </div>
                <div class="col-md-6">
                  <p><strong>授課教師：</strong><?= e($section['teacher_name'] ?? '-') ?></p>
                  <p><strong>教室：</strong><?= e($section['classroom'] ?? '-') ?></p>
                </div>
              </div>
            </div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card h-100">
            <div class="card-header"><i class="bi bi-speedometer2 me-2"></i>快捷操作</div>
            <div class="card-body">
              <a href="attendance_entry.php?section_id=<?= $sectionId ?>" class="btn btn-success w-100 mb-2">
                <i class="bi bi-clipboard-check me-1"></i>點名作業
              </a>
              <a href="grades_entry.php?section_id=<?= $sectionId ?>" class="btn btn-warning w-100">
                <i class="bi bi-journal-text me-1"></i>成績輸入
              </a>
            </div>
          </div>
        </div>
      </div>
      
      <!-- 學生名單 -->
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <span><i class="bi bi-people me-2"></i>學生名單</span>
          <div>
            <?php if ($canEdit): ?>
            <button type="button" class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#addStudentModal">
              <i class="bi bi-person-plus me-1"></i>加選學生
            </button>
            <?php endif; ?>
          </div>
        </div>
        <div class="card-body p-0">
          <div class="table-responsive">
            <table class="table table-hover mb-0">
              <thead>
                <tr class="table-primary">
                  <th width="50">#</th>
                  <th>學號</th>
                  <th>姓名</th>
                  <th>國籍</th>
                  <th class="text-center">總分</th>
                  <th class="text-center">等第</th>
                  <th class="text-center">出席率</th>
                  <th class="text-center">狀態</th>
                  <?php if ($canEdit): ?><th width="150">操作</th><?php endif; ?>
                </tr>
              </thead>
              <tbody>
                <?php if (empty($students)): ?>
                <tr><td colspan="<?= $canEdit ? 9 : 8 ?>" class="text-center text-muted py-4">此班級尚無學生</td></tr>
                <?php else: ?>
                <?php $i = 1; foreach ($students as $s): ?>
                <?php $est = get_enrollment_status($s['enrollment_status']); ?>
                <tr class="<?= $s['enrollment_status'] != 'active' ? 'table-secondary' : '' ?>">
                  <td><?= $i++ ?></td>
                  <td><?= e($s['student_no']) ?></td>
                  <td>
                    <?= e($s['name']) ?>
                    <?php if ($s['chinese_name']): ?><br><small class="text-muted"><?= e($s['chinese_name']) ?></small><?php endif; ?>
                  </td>
                  <td><?= e($s['nationality']) ?></td>
                  <td class="text-center fw-bold"><?= $s['final_score'] !== null ? number_format($s['final_score'], 1) : '-' ?></td>
                  <td class="text-center">
                    <?php if ($s['final_grade']): ?>
                    <span class="badge bg-<?= get_grade_class($s['final_grade']) ?>"><?= $s['final_grade'] ?></span>
                    <?php else: ?>-<?php endif; ?>
                  </td>
                  <td class="text-center"><?= $s['attendance_rate'] !== null ? $s['attendance_rate'] . '%' : '-' ?></td>
                  <td class="text-center"><span class="badge bg-<?= $est['class'] ?>"><?= $est['name'] ?></span></td>
                  <?php if ($canEdit): ?>
                  <td>
                    <?php if ($s['enrollment_status'] === 'active'): ?>
                    <?php if (!empty($otherSections)): ?>
                    <button class="btn btn-sm btn-outline-primary" onclick="showTransfer(<?= $s['enrollment_id'] ?>, '<?= e($s['name']) ?>')">調班</button>
                    <?php endif; ?>
                    <button class="btn btn-sm btn-outline-danger" onclick="dropStudent(<?= $s['enrollment_id'] ?>, '<?= e($s['name']) ?>')">退選</button>
                    <?php endif; ?>
                  </td>
                  <?php endif; ?>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </div>
  </main>

  <!-- 加選學生 Modal -->
  <div class="modal fade" id="addStudentModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title">加選學生</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <div class="input-group mb-3">
            <input type="text" id="searchStudent" class="form-control" placeholder="搜尋學號或姓名...">
            <button class="btn btn-primary" type="button" onclick="searchStudents()">搜尋</button>
          </div>
          <div id="searchResults"></div>
        </div>
      </div>
    </div>
  </div>

  <!-- 調班 Modal -->
  <div class="modal fade" id="transferModal" tabindex="-1">
    <div class="modal-dialog">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title">調班 - <span id="transferStudentName"></span></h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <input type="hidden" id="transferEnrollmentId">
          <div class="mb-3">
            <label class="form-label">目標班級</label>
            <select id="targetSectionId" class="form-select">
              <?php foreach ($otherSections as $os): ?>
              <option value="<?= $os['section_id'] ?>"><?= e($os['section_name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="mb-3">
            <label class="form-label">調班原因</label>
            <input type="text" id="transferReason" class="form-control" placeholder="選填">
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">取消</button>
          <button type="button" class="btn btn-primary" onclick="doTransfer()">確認調班</button>
        </div>
      </div>
    </div>
  </div>

  <?php include_once "temp_footer.php"; ?>

  <script>
    showSidebar("sections", "sections_list");
    
    const sectionId = <?= $sectionId ?>;
    const levelCode = '<?= e($section['level_code']) ?>';

    function searchStudents() {
      const keyword = document.getElementById('searchStudent').value.trim();
      if (!keyword) return;
      
      ajaxPost('api/students_api.php', {action: 'search', keyword: keyword, level_code: levelCode, exclude_section: sectionId}, function(res) {
        if (res.success && res.data.length > 0) {
          let html = '<table class="table table-sm table-hover"><thead><tr><th>學號</th><th>姓名</th><th>程度</th><th></th></tr></thead><tbody>';
          res.data.forEach(s => {
            html += `<tr>
              <td>${s.student_no}</td>
              <td>${s.name}</td>
              <td><span class="badge bg-info">${s.level_code}</span></td>
              <td><button class="btn btn-sm btn-primary" onclick="addStudent(${s.student_id})">加選</button></td>
            </tr>`;
          });
          html += '</tbody></table>';
          document.getElementById('searchResults').innerHTML = html;
        } else {
          document.getElementById('searchResults').innerHTML = '<p class="text-muted text-center">找不到符合的學生</p>';
        }
      });
    }

    function addStudent(studentId) {
      if (!confirm('確定要加選此學生嗎？')) return;
      
      ajaxPost('api/sections_api.php', {action: 'add_student', section_id: sectionId, student_id: studentId}, function(res) {
        if (res.success) {
          showToast('已加選');
          location.reload();
        } else {
          alert(res.message);
        }
      });
    }

    function showTransfer(enrollmentId, name) {
      document.getElementById('transferEnrollmentId').value = enrollmentId;
      document.getElementById('transferStudentName').textContent = name;
      new bootstrap.Modal(document.getElementById('transferModal')).show();
    }

    function doTransfer() {
      const enrollmentId = document.getElementById('transferEnrollmentId').value;
      const targetSection = document.getElementById('targetSectionId').value;
      const reason = document.getElementById('transferReason').value;
      
      ajaxPost('api/sections_api.php', {
        action: 'transfer',
        enrollment_id: enrollmentId,
        target_section_id: targetSection,
        reason: reason
      }, function(res) {
        if (res.success) {
          showToast('調班成功');
          location.reload();
        } else {
          alert(res.message);
        }
      });
    }

    function dropStudent(enrollmentId, name) {
      const reason = prompt('請輸入退選原因（' + name + '）：', '學生申請');
      if (reason === null) return;
      
      ajaxPost('api/sections_api.php', {action: 'drop', enrollment_id: enrollmentId, reason: reason}, function(res) {
        if (res.success) {
          showToast('已退選');
          location.reload();
        } else {
          alert(res.message);
        }
      });
    }
  </script>
</body>
</html>
