<?php
/**
 * ISLMS - 成績輸入
 */
include_once 'src/function.php';
check_login();

$sectionId = intval($_GET['section_id'] ?? 0);
if (!$sectionId) {
    set_flash('error', '請選擇班級');
    redirect('grades_list.php');
}

// 取得班級資訊
$stmt = $conn->prepare("SELECT s.*, o.semester, o.offering_id, c.course_name, c.level_code 
                        FROM sections s
                        JOIN offerings o ON s.offering_id = o.offering_id
                        JOIN courses c ON o.course_id = c.course_id
                        WHERE s.section_id = ?");
$stmt->execute([$sectionId]);
$section = $stmt->fetch();

if (!$section) {
    set_flash('error', '班級不存在');
    redirect('grades_list.php');
}

// 教師權限檢查
if ($_SESSION['user_role'] === 'teacher') {
    $stmt = $conn->prepare("SELECT 1 FROM sections s JOIN offerings o ON s.offering_id = o.offering_id WHERE s.section_id = ? AND (s.teacher_id = ? OR o.teacher_id = ?)");
    $stmt->execute([$sectionId, $_SESSION['user_id'], $_SESSION['user_id']]);
    if (!$stmt->fetch()) {
        set_flash('error', '您沒有權限操作此班級');
        redirect('grades_list.php');
    }
}

// 取得評分項目
$stmt = $conn->prepare("SELECT * FROM grade_items WHERE offering_id = ? ORDER BY sort_order");
$stmt->execute([$section['offering_id']]);
$gradeItems = $stmt->fetchAll();

// 如果沒有評分項目，建立預設項目
if (empty($gradeItems)) {
    $defaults = [
        ['平時成績', 30, 1],
        ['期中考', 30, 2],
        ['期末考', 40, 3]
    ];
    foreach ($defaults as $d) {
        $stmt = $conn->prepare("INSERT INTO grade_items (offering_id, item_name, weight, sort_order) VALUES (?, ?, ?, ?)");
        $stmt->execute([$section['offering_id'], $d[0], $d[1], $d[2]]);
    }
    $stmt = $conn->prepare("SELECT * FROM grade_items WHERE offering_id = ? ORDER BY sort_order");
    $stmt->execute([$section['offering_id']]);
    $gradeItems = $stmt->fetchAll();
}

// 取得學生名單與成績
$sql = "SELECT s.*, e.enrollment_id, e.final_score, e.final_grade, e.attendance_rate
        FROM students s
        JOIN enrollments e ON s.student_id = e.student_id
        WHERE e.section_id = ? AND e.enrollment_status = 'active'
        ORDER BY s.student_no";
$stmt = $conn->prepare($sql);
$stmt->execute([$sectionId]);
$students = $stmt->fetchAll();

// 取得所有成績記錄
$grades = [];
$stmt = $conn->prepare("SELECT g.* FROM grades g 
                        JOIN enrollments e ON g.enrollment_id = e.enrollment_id 
                        WHERE e.section_id = ?");
$stmt->execute([$sectionId]);
foreach ($stmt->fetchAll() as $g) {
    $grades[$g['enrollment_id']][$g['item_id']] = $g;
}
?>
<!DOCTYPE html>
<html lang="zh-TW">
<head>
  <meta charset="utf-8">
  <meta content="width=device-width, initial-scale=1.0" name="viewport">
  <title>成績輸入 - <?= e($section['section_name']) ?> - <?= APP_NAME ?></title>
  <?php include_once "temp_head.php"; ?>
</head>
<body>
  <?php include_once "temp_header.php"; ?>
  <?php include_once "temp_sidebar.php"; ?>

  <main id="main" class="main">
    <div class="pagetitle">
      <h1><?= e($section['course_name']) ?> - <?= e($section['section_name']) ?></h1>
      <nav>
        <ol class="breadcrumb">
          <li class="breadcrumb-item"><a href="index.php">Home</a></li>
          <li class="breadcrumb-item"><a href="grades_list.php">成績管理</a></li>
          <li class="breadcrumb-item active"><?= e($section['section_name']) ?></li>
        </ol>
      </nav>
    </div>

    <div class="container-fluid">
      <?php show_flash(); ?>
      
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <span><i class="bi bi-journal-text me-2"></i>成績輸入</span>
          <div>
            <button type="button" class="btn btn-sm btn-success" onclick="calculateAll()">
              <i class="bi bi-calculator me-1"></i>重新計算
            </button>
            <button type="button" class="btn btn-sm btn-primary" onclick="saveGrades()">
              <i class="bi bi-save me-1"></i>儲存
            </button>
          </div>
        </div>
        <div class="card-body p-0">
          <?php if (empty($students)): ?>
          <div class="text-center text-muted py-5">此班級尚無學生</div>
          <?php else: ?>
          <form id="gradeForm">
            <div class="table-responsive">
              <table class="table table-hover table-bordered mb-0">
                <thead>
                  <tr class="table-primary">
                    <th width="50">#</th>
                    <th width="100">學號</th>
                    <th>姓名</th>
                    <?php foreach ($gradeItems as $item): ?>
                    <th width="100" class="text-center">
                      <?= e($item['item_name']) ?><br>
                      <small class="text-muted">(<?= $item['weight'] ?>%)</small>
                    </th>
                    <?php endforeach; ?>
                    <th width="80" class="text-center">總分</th>
                    <th width="60" class="text-center">等第</th>
                    <th width="80" class="text-center">出席率</th>
                  </tr>
                </thead>
                <tbody>
                  <?php $i = 1; foreach ($students as $s): ?>
                  <tr data-enrollment="<?= $s['enrollment_id'] ?>">
                    <td><?= $i++ ?></td>
                    <td><?= e($s['student_no']) ?></td>
                    <td>
                      <?= e($s['name']) ?>
                      <?php if ($s['chinese_name']): ?><br><small class="text-muted"><?= e($s['chinese_name']) ?></small><?php endif; ?>
                    </td>
                    <?php foreach ($gradeItems as $item): ?>
                    <?php $g = $grades[$s['enrollment_id']][$item['item_id']] ?? null; ?>
                    <td class="text-center">
                      <input type="number" class="form-control form-control-sm text-center grade-input" 
                             name="grades[<?= $s['enrollment_id'] ?>][<?= $item['item_id'] ?>]"
                             data-weight="<?= $item['weight'] ?>"
                             min="0" max="100" step="0.1"
                             value="<?= $g['score'] ?? '' ?>"
                             onchange="calcRow(this)">
                    </td>
                    <?php endforeach; ?>
                    <td class="text-center fw-bold">
                      <span class="total-score"><?= $s['final_score'] !== null ? number_format($s['final_score'], 1) : '-' ?></span>
                    </td>
                    <td class="text-center">
                      <span class="grade-letter badge bg-<?= get_grade_class($s['final_grade']) ?>"><?= $s['final_grade'] ?? '-' ?></span>
                    </td>
                    <td class="text-center">
                      <span class="attendance-rate"><?= $s['attendance_rate'] !== null ? $s['attendance_rate'] . '%' : '-' ?></span>
                    </td>
                  </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          </form>
          <?php endif; ?>
        </div>
      </div>
      
      <div class="card mt-3">
        <div class="card-body py-2">
          <small class="text-muted">
            <strong>等第：</strong>
            <span class="badge bg-success">A (80-100)</span>
            <span class="badge bg-primary">B (70-79)</span>
            <span class="badge bg-warning">C (60-69)</span>
            <span class="badge bg-danger">D (&lt;60)</span>
            <span class="ms-3">快捷鍵：Ctrl+S 儲存</span>
          </small>
        </div>
      </div>
    </div>
  </main>

  <?php include_once "temp_footer.php"; ?>

  <script>
    showSidebar("grades", "grades_list");
    
    const sectionId = <?= $sectionId ?>;

    function calcRow(input) {
      const row = input.closest('tr');
      const inputs = row.querySelectorAll('.grade-input');
      let total = 0;
      let hasScore = false;
      
      inputs.forEach(inp => {
        const score = parseFloat(inp.value);
        const weight = parseFloat(inp.dataset.weight);
        if (!isNaN(score)) {
          total += score * weight / 100;
          hasScore = true;
        }
      });
      
      if (hasScore) {
        row.querySelector('.total-score').textContent = total.toFixed(1);
        const grade = total >= 80 ? 'A' : total >= 70 ? 'B' : total >= 60 ? 'C' : 'D';
        const gradeClass = grade === 'A' ? 'success' : grade === 'B' ? 'primary' : grade === 'C' ? 'warning' : 'danger';
        row.querySelector('.grade-letter').className = 'grade-letter badge bg-' + gradeClass;
        row.querySelector('.grade-letter').textContent = grade;
      }
    }

    function saveGrades() {
      const formData = new FormData(document.getElementById('gradeForm'));
      formData.append('action', 'save');
      formData.append('section_id', sectionId);
      formData.append('csrfToken', csrfToken);
      
      fetch('api/grades_api.php', { method: 'POST', body: formData })
        .then(r => r.json())
        .then(res => {
          if (res.success) showToast(res.message);
          else alert(res.message);
        });
    }

    function calculateAll() {
      ajaxPost('api/grades_api.php', {action: 'calculate_all', section_id: sectionId}, function(res) {
        if (res.success) {
          showToast('已重新計算');
          location.reload();
        } else {
          alert(res.message);
        }
      });
    }

    document.addEventListener('keydown', function(e) {
      if (e.ctrlKey && e.key === 's') { e.preventDefault(); saveGrades(); }
    });
  </script>
</body>
</html>
