<?php
/**
 * ISLMS - 點名輸入
 */
include_once 'src/function.php';
check_login();

$sectionId = intval($_GET['section_id'] ?? 0);
$classDate = $_GET['date'] ?? date('Y-m-d');

if (!$sectionId) {
    set_flash('error', '請選擇班級');
    redirect('attendance_list.php');
}

// 取得班級資訊
$stmt = $conn->prepare("SELECT s.*, o.semester, o.start_date, o.end_date, c.course_name, c.level_code 
                        FROM sections s
                        JOIN offerings o ON s.offering_id = o.offering_id
                        JOIN courses c ON o.course_id = c.course_id
                        WHERE s.section_id = ?");
$stmt->execute([$sectionId]);
$section = $stmt->fetch();

if (!$section) {
    set_flash('error', '班級不存在');
    redirect('attendance_list.php');
}

// 教師權限檢查
if ($_SESSION['user_role'] === 'teacher') {
    $stmt = $conn->prepare("SELECT 1 FROM sections s JOIN offerings o ON s.offering_id = o.offering_id WHERE s.section_id = ? AND (s.teacher_id = ? OR o.teacher_id = ?)");
    $stmt->execute([$sectionId, $_SESSION['user_id'], $_SESSION['user_id']]);
    if (!$stmt->fetch()) {
        set_flash('error', '您沒有權限操作此班級');
        redirect('attendance_list.php');
    }
}

// 取得學生名單
$stmt = $conn->prepare("SELECT s.*, e.enrollment_id FROM students s
                        JOIN enrollments e ON s.student_id = e.student_id
                        WHERE e.section_id = ? AND e.enrollment_status = 'active'
                        ORDER BY s.student_no");
$stmt->execute([$sectionId]);
$students = $stmt->fetchAll();

// 取得現有點名記錄
$stmt = $conn->prepare("SELECT * FROM attendance WHERE section_id = ? AND class_date = ?");
$stmt->execute([$sectionId, $classDate]);
$records = [];
foreach ($stmt->fetchAll() as $r) {
    $records[$r['student_id']] = $r;
}

// 檢查是否已鎖定
$stmt = $conn->prepare("SELECT is_locked FROM attendance WHERE section_id = ? AND class_date = ? AND is_locked = 1 LIMIT 1");
$stmt->execute([$sectionId, $classDate]);
$isLocked = (bool)$stmt->fetch();

// 取得已點名日期
$stmt = $conn->prepare("SELECT DISTINCT class_date FROM attendance WHERE section_id = ? ORDER BY class_date DESC");
$stmt->execute([$sectionId]);
$dates = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="zh-TW">
<head>
    <meta charset="utf-8">
    <meta content="width=device-width, initial-scale=1.0" name="viewport">
    <title>點名 - <?= e($section['section_name']) ?> - <?= APP_NAME ?></title>
    <?php include_once "temp_head.php"; ?>
</head>
<body>
    <?php include_once "temp_header.php"; ?>
    <?php include_once "temp_sidebar.php"; ?>

    <main id="main" class="main">
        <div class="pagetitle">
            <h1><?= e($section['course_name']) ?> - <?= e($section['section_name']) ?></h1>
            <nav>
                <ol class="breadcrumb">
                    <li class="breadcrumb-item"><a href="index.php">Home</a></li>
                    <li class="breadcrumb-item"><a href="attendance_list.php">點名作業</a></li>
                    <li class="breadcrumb-item active"><?= e($section['section_name']) ?></li>
                </ol>
            </nav>
        </div>

        <div class="container-fluid">
            <?php show_flash(); ?>
            
            <div class="row">
                <!-- 日期選擇 -->
                <div class="col-md-3 mb-4">
                    <div class="card">
                        <div class="card-header"><i class="bi bi-calendar3 me-2"></i>點名日期</div>
                        <div class="card-body">
                            <div class="mb-3">
                                <input type="date" id="selectDate" class="form-control" value="<?= e($classDate) ?>">
                            </div>
                            <button type="button" class="btn btn-primary w-100 mb-3" onclick="loadDate()">
                                <i class="bi bi-search me-1"></i>查看/新增
                            </button>
                            
                            <?php if (!empty($dates)): ?>
                            <hr>
                            <div class="small text-muted mb-2">已點名日期：</div>
                            <div class="list-group list-group-flush" style="max-height: 250px; overflow-y: auto;">
                                <?php foreach ($dates as $d): ?>
                                <a href="?section_id=<?= $sectionId ?>&date=<?= $d['class_date'] ?>" 
                                   class="list-group-item list-group-item-action <?= $d['class_date'] == $classDate ? 'active' : '' ?>">
                                    <?= format_date($d['class_date'], 'm/d (D)') ?>
                                </a>
                                <?php endforeach; ?>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <!-- 點名表 -->
                <div class="col-md-9">
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <span>
                                <i class="bi bi-clipboard-check me-2"></i>
                                <?= format_date($classDate, 'Y年m月d日') ?> 點名表
                                <?php if ($isLocked): ?>
                                <span class="badge bg-warning ms-2"><i class="bi bi-lock-fill"></i> 已鎖定</span>
                                <?php endif; ?>
                            </span>
                            <div>
                                <?php if (!$isLocked): ?>
                                <button type="button" class="btn btn-sm btn-success" onclick="setAllPresent()">
                                    <i class="bi bi-check-all me-1"></i>全部出席
                                </button>
                                <button type="button" class="btn btn-sm btn-primary" onclick="saveAttendance()">
                                    <i class="bi bi-save me-1"></i>儲存
                                </button>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="card-body p-0">
                            <?php if (empty($students)): ?>
                            <div class="text-center text-muted py-5">此班級尚無學生</div>
                            <?php else: ?>
                            <form id="attendanceForm">
                                <div class="table-responsive">
                                    <table class="table table-hover mb-0">
                                        <thead>
                                            <tr>
                                                <th width="50">#</th>
                                                <th>學號</th>
                                                <th>姓名</th>
                                                <th width="350">出席狀態</th>
                                                <th>備註</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php $i = 1; foreach ($students as $s): ?>
                                            <?php $rec = $records[$s['student_id']] ?? null; $curStatus = $rec['attendance_status'] ?? 'P'; ?>
                                            <tr>
                                                <td><?= $i++ ?></td>
                                                <td><?= e($s['student_no']) ?></td>
                                                <td>
                                                    <?= e($s['name']) ?>
                                                    <?php if ($s['chinese_name']): ?><br><small class="text-muted"><?= e($s['chinese_name']) ?></small><?php endif; ?>
                                                </td>
                                                <td>
                                                    <div class="btn-group btn-group-sm btn-group-attendance" data-student="<?= $s['student_id'] ?>">
                                                        <?php
                                                        $statuses = [
                                                            'P' => ['出席', 'success'], 'L' => ['遲到', 'warning'], 'A' => ['缺席', 'danger'],
                                                            'E' => ['請假', 'info'], 'O' => ['公假', 'primary'], 'S' => ['病假', 'secondary']
                                                        ];
                                                        foreach ($statuses as $code => $info):
                                                        ?>
                                                        <button type="button" class="btn btn-<?= $curStatus == $code ? '' : 'outline-' ?><?= $info[1] ?>"
                                                                data-status="<?= $code ?>" <?= $isLocked ? 'disabled' : '' ?>
                                                                onclick="setStatus(<?= $s['student_id'] ?>, '<?= $code ?>', this)"><?= $info[0] ?></button>
                                                        <?php endforeach; ?>
                                                    </div>
                                                    <input type="hidden" name="attendance[<?= $s['student_id'] ?>][status]" value="<?= $curStatus ?>">
                                                </td>
                                                <td>
                                                    <input type="text" class="form-control form-control-sm" 
                                                           name="attendance[<?= $s['student_id'] ?>][note]"
                                                           value="<?= e($rec['note'] ?? '') ?>" <?= $isLocked ? 'readonly' : '' ?>>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </form>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <div class="card mt-3">
                        <div class="card-body py-2">
                            <small class="text-muted">
                                <strong>狀態：</strong>
                                <span class="badge bg-success">P 出席</span>
                                <span class="badge bg-warning">L 遲到</span>
                                <span class="badge bg-danger">A 缺席</span>
                                <span class="badge bg-info">E 請假</span>
                                <span class="badge bg-primary">O 公假</span>
                                <span class="badge bg-secondary">S 病假</span>
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <?php include_once "temp_footer.php"; ?>

    <script>
        showSidebar("attendance", "attendance_list");
        
        const sectionId = <?= $sectionId ?>;
        const classDate = '<?= $classDate ?>';
        const isLocked = <?= $isLocked ? 'true' : 'false' ?>;

        function loadDate() {
            const date = document.getElementById('selectDate').value;
            if (date) window.location.href = '?section_id=' + sectionId + '&date=' + date;
        }

        function setStatus(studentId, status, btn) {
            if (isLocked) return;
            const group = btn.parentElement;
            group.querySelectorAll('button').forEach(b => {
                const st = b.dataset.status;
                const color = b.className.match(/btn-(?:outline-)?(success|warning|danger|info|primary|secondary)/)[1];
                b.className = 'btn btn-' + (st === status ? '' : 'outline-') + color;
            });
            document.querySelector(`input[name="attendance[${studentId}][status]"]`).value = status;
        }

        function setAllPresent() {
            if (isLocked) return;
            document.querySelectorAll('.btn-group-attendance').forEach(group => {
                const studentId = group.dataset.student;
                const btn = group.querySelector('[data-status="P"]');
                setStatus(studentId, 'P', btn);
            });
            showToast('已設定全部出席');
        }

        function saveAttendance() {
            if (isLocked) { alert('點名記錄已鎖定'); return; }
            
            const formData = new FormData(document.getElementById('attendanceForm'));
            formData.append('action', 'save');
            formData.append('section_id', sectionId);
            formData.append('class_date', classDate);
            formData.append('csrfToken', csrfToken);
            
            fetch('api/attendance_api.php', { method: 'POST', body: formData })
                .then(r => r.json())
                .then(res => {
                    if (res.success) showToast(res.message);
                    else alert(res.message);
                });
        }

        document.addEventListener('keydown', function(e) {
            if (e.ctrlKey && e.key === 's') { e.preventDefault(); saveAttendance(); }
        });
    </script>
</body>
</html>
