<?php
/**
 * ISLMS - 班級 API
 */
include_once '../src/function.php';
check_login();

header('Content-Type: application/json; charset=utf-8');

$action = $_POST['action'] ?? '';

switch ($action) {
    case 'create_section':
        if (!has_role(['admin', 'staff'])) json_error('權限不足');
        
        $offeringId = intval($_POST['offering_id'] ?? 0);
        $sectionName = trim($_POST['section_name'] ?? '');
        $teacherId = intval($_POST['teacher_id'] ?? 0) ?: null;
        $classroom = trim($_POST['classroom'] ?? '') ?: null;
        $maxStudents = intval($_POST['max_students'] ?? 12);
        
        if (!$offeringId || !$sectionName) json_error('參數錯誤');
        
        $stmt = $conn->prepare("INSERT INTO sections (offering_id, section_name, teacher_id, classroom, max_students, section_status, created_by) VALUES (?, ?, ?, ?, ?, 'active', ?)");
        $stmt->execute([$offeringId, $sectionName, $teacherId, $classroom, $maxStudents, $_SESSION['user_account']]);
        
        write_log('create', 'sections', $conn->lastInsertId(), '新增班級: ' . $sectionName);
        json_success('班級已新增');
        break;
        
    case 'auto_assign':
        if (!has_role(['admin', 'staff'])) json_error('權限不足');
        
        $offeringId = intval($_POST['offering_id'] ?? 0);
        if (!$offeringId) json_error('參數錯誤');
        
        // 取得開班設定
        $stmt = $conn->prepare("SELECT o.*, c.level_code FROM offerings o JOIN courses c ON o.course_id = c.course_id WHERE o.offering_id = ?");
        $stmt->execute([$offeringId]);
        $offering = $stmt->fetch();
        if (!$offering) json_error('開班不存在');
        
        $maxPerSection = $offering['max_students_per_section'] ?: 12;
        
        // 取得未分班學生
        $stmt = $conn->prepare("SELECT s.student_id FROM students s 
                                WHERE s.level_code = ? AND s.status = 1 
                                AND s.student_id NOT IN (
                                    SELECT e.student_id FROM enrollments e 
                                    JOIN sections sec ON e.section_id = sec.section_id 
                                    WHERE sec.offering_id = ? AND e.enrollment_status = 'active'
                                )
                                ORDER BY s.student_no");
        $stmt->execute([$offering['level_code'], $offeringId]);
        $students = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (empty($students)) json_error('沒有待分班的學生');
        
        // 取得現有班級
        $stmt = $conn->prepare("SELECT section_id, section_name, max_students, current_count FROM sections WHERE offering_id = ? AND section_status = 'active' ORDER BY section_name");
        $stmt->execute([$offeringId]);
        $sections = $stmt->fetchAll();
        
        // 如果沒有班級，自動建立
        if (empty($sections)) {
            $numSections = ceil(count($students) / $maxPerSection);
            for ($i = 1; $i <= $numSections; $i++) {
                $sectionName = chr(64 + $i) . '班'; // A班, B班, ...
                $stmt = $conn->prepare("INSERT INTO sections (offering_id, section_name, max_students, section_status, created_by) VALUES (?, ?, ?, 'active', ?)");
                $stmt->execute([$offeringId, $sectionName, $maxPerSection, $_SESSION['user_account']]);
                $sections[] = ['section_id' => $conn->lastInsertId(), 'section_name' => $sectionName, 'max_students' => $maxPerSection, 'current_count' => 0];
            }
        }
        
        // 分配學生
        $conn->beginTransaction();
        try {
            $sectionIndex = 0;
            $assigned = 0;
            
            foreach ($students as $studentId) {
                // 找有空位的班級
                while ($sectionIndex < count($sections)) {
                    $sec = $sections[$sectionIndex];
                    if ($sec['current_count'] < $sec['max_students']) {
                        break;
                    }
                    $sectionIndex++;
                }
                
                if ($sectionIndex >= count($sections)) {
                    // 需要新增班級
                    $newName = chr(65 + count($sections)) . '班';
                    $stmt = $conn->prepare("INSERT INTO sections (offering_id, section_name, max_students, section_status, created_by) VALUES (?, ?, ?, 'active', ?)");
                    $stmt->execute([$offeringId, $newName, $maxPerSection, $_SESSION['user_account']]);
                    $sections[] = ['section_id' => $conn->lastInsertId(), 'section_name' => $newName, 'max_students' => $maxPerSection, 'current_count' => 0];
                }
                
                $sectionId = $sections[$sectionIndex]['section_id'];
                
                // 新增選課記錄
                $stmt = $conn->prepare("INSERT INTO enrollments (student_id, section_id, enrollment_status, enrolled_at, created_by) VALUES (?, ?, 'active', NOW(), ?)");
                $stmt->execute([$studentId, $sectionId, $_SESSION['user_account']]);
                
                // 更新班級人數
                $stmt = $conn->prepare("UPDATE sections SET current_count = current_count + 1 WHERE section_id = ?");
                $stmt->execute([$sectionId]);
                $sections[$sectionIndex]['current_count']++;
                
                $assigned++;
            }
            
            $conn->commit();
            write_log('auto_assign', 'offerings', $offeringId, "自動編班完成，分配 $assigned 人");
            json_success("已分配 $assigned 位學生");
        } catch (Exception $e) {
            $conn->rollBack();
            json_error('編班失敗：' . $e->getMessage());
        }
        break;
        
    case 'add_student':
        if (!has_role(['admin', 'staff'])) json_error('權限不足');
        
        $sectionId = intval($_POST['section_id'] ?? 0);
        $studentId = intval($_POST['student_id'] ?? 0);
        
        if (!$sectionId || !$studentId) json_error('參數錯誤');
        
        // 檢查是否已選課
        $stmt = $conn->prepare("SELECT 1 FROM enrollments e JOIN sections s ON e.section_id = s.section_id 
                                WHERE e.student_id = ? AND s.offering_id = (SELECT offering_id FROM sections WHERE section_id = ?) AND e.enrollment_status = 'active'");
        $stmt->execute([$studentId, $sectionId]);
        if ($stmt->fetch()) json_error('此學生已在本課程中');
        
        $stmt = $conn->prepare("INSERT INTO enrollments (student_id, section_id, enrollment_status, enrolled_at, created_by) VALUES (?, ?, 'active', NOW(), ?)");
        $stmt->execute([$studentId, $sectionId, $_SESSION['user_account']]);
        
        $stmt = $conn->prepare("UPDATE sections SET current_count = current_count + 1 WHERE section_id = ?");
        $stmt->execute([$sectionId]);
        
        write_log('add_student', 'enrollments', $conn->lastInsertId(), '加選學生');
        json_success('已加選');
        break;
        
    case 'transfer':
        if (!has_role(['admin', 'staff'])) json_error('權限不足');
        
        $enrollmentId = intval($_POST['enrollment_id'] ?? 0);
        $targetSectionId = intval($_POST['target_section_id'] ?? 0);
        $reason = trim($_POST['reason'] ?? '');
        
        if (!$enrollmentId || !$targetSectionId) json_error('參數錯誤');
        
        // 取得原選課記錄
        $stmt = $conn->prepare("SELECT * FROM enrollments WHERE enrollment_id = ?");
        $stmt->execute([$enrollmentId]);
        $enrollment = $stmt->fetch();
        if (!$enrollment) json_error('選課記錄不存在');
        
        $conn->beginTransaction();
        try {
            // 更新選課記錄
            $stmt = $conn->prepare("UPDATE enrollments SET section_id = ? WHERE enrollment_id = ?");
            $stmt->execute([$targetSectionId, $enrollmentId]);
            
            // 更新班級人數
            $stmt = $conn->prepare("UPDATE sections SET current_count = current_count - 1 WHERE section_id = ?");
            $stmt->execute([$enrollment['section_id']]);
            $stmt = $conn->prepare("UPDATE sections SET current_count = current_count + 1 WHERE section_id = ?");
            $stmt->execute([$targetSectionId]);
            
            // 記錄異動
            $stmt = $conn->prepare("INSERT INTO enrollment_history (enrollment_id, change_type, from_section_id, to_section_id, reason, created_by) VALUES (?, 'transfer', ?, ?, ?, ?)");
            $stmt->execute([$enrollmentId, $enrollment['section_id'], $targetSectionId, $reason, $_SESSION['user_account']]);
            
            $conn->commit();
            write_log('transfer', 'enrollments', $enrollmentId, '調班');
            json_success('調班成功');
        } catch (Exception $e) {
            $conn->rollBack();
            json_error('調班失敗');
        }
        break;
        
    case 'drop':
        if (!has_role(['admin', 'staff'])) json_error('權限不足');
        
        $enrollmentId = intval($_POST['enrollment_id'] ?? 0);
        $reason = trim($_POST['reason'] ?? '');
        
        if (!$enrollmentId) json_error('參數錯誤');
        
        $stmt = $conn->prepare("SELECT * FROM enrollments WHERE enrollment_id = ?");
        $stmt->execute([$enrollmentId]);
        $enrollment = $stmt->fetch();
        if (!$enrollment) json_error('選課記錄不存在');
        
        $conn->beginTransaction();
        try {
            $stmt = $conn->prepare("UPDATE enrollments SET enrollment_status = 'dropped', dropped_at = NOW() WHERE enrollment_id = ?");
            $stmt->execute([$enrollmentId]);
            
            $stmt = $conn->prepare("UPDATE sections SET current_count = current_count - 1 WHERE section_id = ?");
            $stmt->execute([$enrollment['section_id']]);
            
            $stmt = $conn->prepare("INSERT INTO enrollment_history (enrollment_id, change_type, from_section_id, reason, created_by) VALUES (?, 'drop', ?, ?, ?)");
            $stmt->execute([$enrollmentId, $enrollment['section_id'], $reason, $_SESSION['user_account']]);
            
            $conn->commit();
            write_log('drop', 'enrollments', $enrollmentId, '退選: ' . $reason);
            json_success('已退選');
        } catch (Exception $e) {
            $conn->rollBack();
            json_error('退選失敗');
        }
        break;
        
    default:
        json_error('未知操作');
}
