<?php
/**
 * ISLMS - 學生管理
 */
require_once __DIR__ . '/config/config.php';

$auth = Auth::getInstance();
if (!$auth->check()) {
    header('Location: ' . APP_URL . '/login.php');
    exit;
}

// 權限檢查
if (!$auth->hasRole(['admin', 'staff'])) {
    setFlash('error', '您沒有權限執行此操作');
    header('Location: ' . APP_URL . '/dashboard.php');
    exit;
}

$studentModel = new StudentModel();
$levelModel = new LevelModel();
$action = $_GET['action'] ?? 'list';

// 處理 AJAX 請求
if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
    header('Content-Type: application/json; charset=utf-8');
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $action = $_POST['action'] ?? '';
        
        if ($action === 'delete') {
            $id = (int) ($_POST['id'] ?? 0);
            if ($studentModel->update($id, ['status' => 0, 'updated_by' => $auth->user()['account']])) {
                echo json_encode(['success' => true, 'message' => '刪除成功']);
            } else {
                echo json_encode(['success' => false, 'message' => '刪除失敗']);
            }
            exit;
        }
    }
    exit;
}

// 頁面處理
switch ($action) {
    case 'create':
    case 'edit':
        $student = null;
        $id = (int) ($_GET['id'] ?? 0);
        
        if ($action === 'edit' && $id) {
            $student = $studentModel->find($id);
            if (!$student) {
                setFlash('error', '學生不存在');
                header('Location: ' . APP_URL . '/students.php');
                exit;
            }
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = [
                'student_no' => trim($_POST['student_no'] ?? ''),
                'chinese_name' => trim($_POST['chinese_name'] ?? ''),
                'name' => trim($_POST['name'] ?? ''),
                'nationality' => trim($_POST['nationality'] ?? ''),
                'gender' => $_POST['gender'] ?? 'M',
                'level_code' => $_POST['level_code'] ?? 'BEG',
                'email' => trim($_POST['email'] ?? ''),
                'phone' => trim($_POST['phone'] ?? ''),
                'student_type' => $_POST['student_type'] ?? 'exchange',
                'note' => trim($_POST['note'] ?? ''),
                'updated_by' => $auth->user()['account']
            ];
            
            // 驗證
            $errors = [];
            if (empty($data['student_no'])) $errors[] = '學號為必填';
            if (empty($data['name'])) $errors[] = '英文姓名為必填';
            if (empty($data['email'])) $errors[] = 'Email 為必填';
            if (!isValidEmail($data['email'])) $errors[] = 'Email 格式不正確';
            
            // 檢查學號是否重複
            if ($studentModel->exists('student_no', $data['student_no'], $id)) {
                $errors[] = '學號已存在';
            }
            
            if (empty($errors)) {
                if ($action === 'edit') {
                    $studentModel->update($id, $data);
                    setFlash('success', '學生資料已更新');
                } else {
                    $data['status'] = 1;
                    $studentModel->create($data);
                    setFlash('success', '學生已新增');
                }
                header('Location: ' . APP_URL . '/students.php');
                exit;
            }
        }
        
        $levels = $levelModel->getActive();
        $pageTitle = ($action === 'edit' ? '編輯' : '新增') . '學生 - ' . APP_NAME;
        $breadcrumb = [
            ['title' => '學生管理', 'url' => url('students.php')],
            ['title' => $action === 'edit' ? '編輯學生' : '新增學生', 'active' => true]
        ];
        
        ob_start();
        ?>
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3 mb-0"><?= $action === 'edit' ? '編輯學生' : '新增學生' ?></h1>
        </div>
        
        <?php if (!empty($errors)): ?>
        <div class="alert alert-danger">
            <ul class="mb-0">
                <?php foreach ($errors as $err): ?>
                <li><?= e($err) ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-body">
                <form method="post" action="">
                    <?= csrfField() ?>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">學號 <span class="text-danger">*</span></label>
                            <input type="text" name="student_no" class="form-control" required
                                   value="<?= e($student['student_no'] ?? $_POST['student_no'] ?? '') ?>">
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">中文程度 <span class="text-danger">*</span></label>
                            <select name="level_code" class="form-select" required>
                                <?php foreach ($levels as $level): ?>
                                <option value="<?= e($level['level_code']) ?>" 
                                    <?= ($student['level_code'] ?? $_POST['level_code'] ?? '') === $level['level_code'] ? 'selected' : '' ?>>
                                    <?= e($level['level_name']) ?> (<?= e($level['level_code']) ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">英文姓名 <span class="text-danger">*</span></label>
                            <input type="text" name="name" class="form-control" required
                                   value="<?= e($student['name'] ?? $_POST['name'] ?? '') ?>">
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">中文姓名</label>
                            <input type="text" name="chinese_name" class="form-control"
                                   value="<?= e($student['chinese_name'] ?? $_POST['chinese_name'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">國籍</label>
                            <input type="text" name="nationality" class="form-control"
                                   value="<?= e($student['nationality'] ?? $_POST['nationality'] ?? '') ?>">
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <label class="form-label">性別 <span class="text-danger">*</span></label>
                            <select name="gender" class="form-select" required>
                                <option value="M" <?= ($student['gender'] ?? $_POST['gender'] ?? 'M') === 'M' ? 'selected' : '' ?>>男</option>
                                <option value="F" <?= ($student['gender'] ?? $_POST['gender'] ?? '') === 'F' ? 'selected' : '' ?>>女</option>
                            </select>
                        </div>
                        
                        <div class="col-md-4 mb-3">
                            <label class="form-label">學生類型</label>
                            <select name="student_type" class="form-select">
                                <option value="exchange" <?= ($student['student_type'] ?? $_POST['student_type'] ?? 'exchange') === 'exchange' ? 'selected' : '' ?>>交換生</option>
                                <option value="degree" <?= ($student['student_type'] ?? $_POST['student_type'] ?? '') === 'degree' ? 'selected' : '' ?>>學位生</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Email <span class="text-danger">*</span></label>
                            <input type="email" name="email" class="form-control" required
                                   value="<?= e($student['email'] ?? $_POST['email'] ?? '') ?>">
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">電話</label>
                            <input type="text" name="phone" class="form-control"
                                   value="<?= e($student['phone'] ?? $_POST['phone'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">備註</label>
                        <textarea name="note" class="form-control" rows="3"><?= e($student['note'] ?? $_POST['note'] ?? '') ?></textarea>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-lg me-1"></i>儲存
                        </button>
                        <a href="<?= url('students.php') ?>" class="btn btn-secondary">取消</a>
                    </div>
                </form>
            </div>
        </div>
        <?php
        $content = ob_get_clean();
        break;
        
    case 'import':
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
            $file = $_FILES['file'];
            
            if ($file['error'] !== UPLOAD_ERR_OK) {
                setFlash('error', '檔案上傳失敗');
                header('Location: ' . APP_URL . '/students.php?action=import');
                exit;
            }
            
            $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            if (!in_array($ext, ['xlsx', 'xls', 'csv'])) {
                setFlash('error', '僅支援 Excel 或 CSV 檔案');
                header('Location: ' . APP_URL . '/students.php?action=import');
                exit;
            }
            
            try {
                // 使用 PhpSpreadsheet 讀取
                require_once ROOT_PATH . '/vendor/autoload.php';
                
                $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file['tmp_name']);
                $sheet = $spreadsheet->getActiveSheet();
                $rows = $sheet->toArray();
                
                // 取得標題列
                $headers = array_shift($rows);
                $headerMap = [];
                foreach ($headers as $i => $h) {
                    $h = strtolower(trim($h));
                    if (strpos($h, '學號') !== false || $h === 'student_id' || $h === 'student_no') {
                        $headerMap['student_no'] = $i;
                    } elseif (strpos($h, '中文姓名') !== false || $h === 'chinese_name') {
                        $headerMap['chinese_name'] = $i;
                    } elseif (strpos($h, '姓名') !== false || $h === 'name') {
                        $headerMap['name'] = $i;
                    } elseif (strpos($h, '國籍') !== false || $h === 'nationality') {
                        $headerMap['nationality'] = $i;
                    } elseif (strpos($h, '性別') !== false || $h === 'gender') {
                        $headerMap['gender'] = $i;
                    } elseif (strpos($h, '程度') !== false || $h === 'level' || $h === 'level_code') {
                        $headerMap['level_code'] = $i;
                    } elseif (strpos($h, 'email') !== false) {
                        $headerMap['email'] = $i;
                    }
                }
                
                // 轉換資料
                $data = [];
                foreach ($rows as $row) {
                    if (empty(array_filter($row))) continue;
                    
                    $item = [];
                    foreach ($headerMap as $field => $index) {
                        $item[$field] = $row[$index] ?? '';
                    }
                    $data[] = $item;
                }
                
                $result = $studentModel->importFromArray($data, $auth->user()['account']);
                
                $msg = "成功匯入 {$result['success']} 筆資料";
                if (!empty($result['errors'])) {
                    $msg .= "，" . count($result['errors']) . " 筆錯誤";
                }
                setFlash('success', $msg);
                
            } catch (Exception $e) {
                setFlash('error', '匯入失敗：' . $e->getMessage());
            }
            
            header('Location: ' . APP_URL . '/students.php');
            exit;
        }
        
        $pageTitle = '匯入學生 - ' . APP_NAME;
        $breadcrumb = [
            ['title' => '學生管理', 'url' => url('students.php')],
            ['title' => '匯入學生', 'active' => true]
        ];
        
        ob_start();
        ?>
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3 mb-0">匯入學生</h1>
        </div>
        
        <div class="card">
            <div class="card-body">
                <form method="post" action="" enctype="multipart/form-data">
                    <?= csrfField() ?>
                    
                    <div class="mb-4">
                        <label class="form-label">選擇檔案 (Excel 或 CSV)</label>
                        <input type="file" name="file" class="form-control" accept=".xlsx,.xls,.csv" required>
                        <div class="form-text">支援 .xlsx, .xls, .csv 格式，檔案大小限制 10MB</div>
                    </div>
                    
                    <div class="alert alert-info">
                        <h6><i class="bi bi-info-circle me-1"></i>匯入格式說明</h6>
                        <p class="mb-1">Excel 檔案第一列為標題列，系統會自動對應以下欄位：</p>
                        <ul class="mb-0">
                            <li><strong>學號</strong> (student_no) - 必填</li>
                            <li><strong>英文姓名</strong> (name) - 必填</li>
                            <li><strong>中文姓名</strong> (chinese_name)</li>
                            <li><strong>國籍</strong> (nationality)</li>
                            <li><strong>性別</strong> (gender) - M/F</li>
                            <li><strong>程度</strong> (level_code) - BEG/BAS/ELE1/ELE2/INT1/INT2/INT3/ADV1</li>
                            <li><strong>Email</strong> (email) - 必填</li>
                        </ul>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-upload me-1"></i>開始匯入
                        </button>
                        <a href="<?= url('students.php') ?>" class="btn btn-secondary">取消</a>
                    </div>
                </form>
            </div>
        </div>
        <?php
        $content = ob_get_clean();
        break;
        
    default: // list
        $keyword = $_GET['keyword'] ?? '';
        $level = $_GET['level'] ?? '';
        $students = $studentModel->search($keyword, $level, 1);
        $levels = $levelModel->getActive();
        
        $pageTitle = '學生管理 - ' . APP_NAME;
        $breadcrumb = [['title' => '學生管理', 'active' => true]];
        
        ob_start();
        ?>
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3 mb-0">學生管理</h1>
            <div>
                <a href="<?= url('students.php?action=import') ?>" class="btn btn-success">
                    <i class="bi bi-upload me-1"></i>匯入
                </a>
                <a href="<?= url('students.php?action=create') ?>" class="btn btn-primary">
                    <i class="bi bi-plus-lg me-1"></i>新增學生
                </a>
            </div>
        </div>
        
        <div class="card mb-4">
            <div class="card-body">
                <form method="get" class="row g-3">
                    <div class="col-md-4">
                        <input type="text" name="keyword" class="form-control" placeholder="搜尋學號、姓名、Email..."
                               value="<?= e($keyword) ?>">
                    </div>
                    <div class="col-md-3">
                        <select name="level" class="form-select">
                            <option value="">全部程度</option>
                            <?php foreach ($levels as $l): ?>
                            <option value="<?= e($l['level_code']) ?>" <?= $level === $l['level_code'] ? 'selected' : '' ?>>
                                <?= e($l['level_name']) ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-auto">
                        <button type="submit" class="btn btn-outline-primary">
                            <i class="bi bi-search me-1"></i>搜尋
                        </button>
                        <a href="<?= url('students.php') ?>" class="btn btn-outline-secondary">清除</a>
                    </div>
                </form>
            </div>
        </div>
        
        <div class="card">
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover data-table mb-0">
                        <thead>
                            <tr>
                                <th>學號</th>
                                <th>姓名</th>
                                <th>國籍</th>
                                <th>程度</th>
                                <th>類型</th>
                                <th>Email</th>
                                <th width="120">操作</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($students as $s): ?>
                            <tr>
                                <td><?= e($s['student_no']) ?></td>
                                <td>
                                    <?= e($s['name']) ?>
                                    <?php if ($s['chinese_name']): ?>
                                    <br><small class="text-muted"><?= e($s['chinese_name']) ?></small>
                                    <?php endif; ?>
                                </td>
                                <td><?= e($s['nationality']) ?></td>
                                <td><span class="badge bg-info"><?= e($s['level_code']) ?></span></td>
                                <td><?= $s['student_type'] === 'degree' ? '學位生' : '交換生' ?></td>
                                <td><small><?= e($s['email']) ?></small></td>
                                <td>
                                    <a href="<?= url('students.php?action=edit&id=' . $s['student_id']) ?>" 
                                       class="btn btn-sm btn-outline-primary" title="編輯">
                                        <i class="bi bi-pencil"></i>
                                    </a>
                                    <button type="button" class="btn btn-sm btn-outline-danger" 
                                            onclick="deleteStudent(<?= $s['student_id'] ?>)" title="刪除">
                                        <i class="bi bi-trash"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <script>
        function deleteStudent(id) {
            if (!confirm('確定要刪除此學生嗎？')) return;
            
            $.post('<?= url('students.php') ?>', {
                action: 'delete',
                id: id,
                <?= CSRF_TOKEN_NAME ?>: csrfToken
            }, function(res) {
                if (res.success) {
                    location.reload();
                } else {
                    alert(res.message);
                }
            });
        }
        </script>
        <?php
        $content = ob_get_clean();
}

$layout = 'main';
include VIEW_PATH . '/layouts/main.php';
