<?php
/**
 * StudentModel - 學生模型
 */
class StudentModel extends Model {
    protected $table = 'students';
    protected $primaryKey = 'student_id';
    protected $fillable = [
        'student_no', 'chinese_name', 'name', 'nationality', 'gender',
        'level_code', 'email', 'phone', 'student_type', 'status', 'note', 'updated_by'
    ];
    
    public function findByStudentNo($studentNo) {
        return $this->findBy('student_no', $studentNo);
    }
    
    public function search($keyword = '', $level = '', $status = '') {
        $conditions = "1=1";
        $params = [];
        
        if ($keyword) {
            $conditions .= " AND (student_no LIKE ? OR chinese_name LIKE ? OR name LIKE ? OR email LIKE ?)";
            $params = array_merge($params, ["%{$keyword}%", "%{$keyword}%", "%{$keyword}%", "%{$keyword}%"]);
        }
        
        if ($level) {
            $conditions .= " AND level_code = ?";
            $params[] = $level;
        }
        
        if ($status !== '') {
            $conditions .= " AND status = ?";
            $params[] = $status;
        }
        
        return $this->where($conditions, $params, 'student_no ASC');
    }
    
    public function getByLevel($levelCode) {
        return $this->where("level_code = ? AND status = 1", [$levelCode], 'student_no ASC');
    }
    
    public function importFromArray($rows, $updatedBy) {
        $db = $this->db;
        $success = 0;
        $errors = [];
        
        $db->beginTransaction();
        try {
            foreach ($rows as $index => $row) {
                $rowNum = $index + 2; // Excel 第一行是標題
                
                // 驗證必填欄位
                if (empty($row['student_no'])) {
                    $errors[] = "第 {$rowNum} 行：學號為必填";
                    continue;
                }
                if (empty($row['name'])) {
                    $errors[] = "第 {$rowNum} 行：英文姓名為必填";
                    continue;
                }
                if (empty($row['email'])) {
                    $errors[] = "第 {$rowNum} 行：Email 為必填";
                    continue;
                }
                
                // 檢查是否已存在
                $existing = $this->findByStudentNo($row['student_no']);
                
                $data = [
                    'student_no' => trim($row['student_no']),
                    'chinese_name' => trim($row['chinese_name'] ?? ''),
                    'name' => trim($row['name']),
                    'nationality' => trim($row['nationality'] ?? ''),
                    'gender' => strtoupper(trim($row['gender'] ?? 'M')),
                    'level_code' => strtoupper(trim($row['level_code'] ?? 'BEG')),
                    'email' => trim($row['email']),
                    'phone' => trim($row['phone'] ?? ''),
                    'student_type' => trim($row['student_type'] ?? 'exchange'),
                    'status' => 1,
                    'updated_by' => $updatedBy
                ];
                
                if ($existing) {
                    // 更新
                    $this->update($existing['student_id'], $data);
                } else {
                    // 新增
                    $this->create($data);
                }
                $success++;
            }
            
            $db->commit();
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
        
        return ['success' => $success, 'errors' => $errors];
    }
    
    public function getStatsByLevel() {
        $sql = "SELECT level_code, COUNT(*) as count 
                FROM {$this->table} 
                WHERE status = 1 
                GROUP BY level_code 
                ORDER BY level_code";
        return $this->db->fetchAll($sql);
    }
    
    public function getStatsByNationality() {
        $sql = "SELECT nationality, COUNT(*) as count 
                FROM {$this->table} 
                WHERE status = 1 
                GROUP BY nationality 
                ORDER BY count DESC 
                LIMIT 10";
        return $this->db->fetchAll($sql);
    }
}
