<?php
/**
 * SectionModel - 班級模型
 */
class SectionModel extends Model {
    protected $table = 'sections';
    protected $primaryKey = 'section_id';
    protected $fillable = [
        'offering_id', 'section_name', 'teacher_id', 'classroom',
        'max_students', 'current_count', 'section_status', 'note', 'updated_by'
    ];
    
    public function getWithDetails($id) {
        $sql = "SELECT s.*, o.semester, o.class_time, o.schedule, o.start_date, o.end_date,
                       c.course_code, c.course_name, c.course_name_en, c.level_code,
                       u.name as teacher_name
                FROM sections s
                LEFT JOIN offerings o ON s.offering_id = o.offering_id
                LEFT JOIN courses c ON o.course_id = c.course_id
                LEFT JOIN users u ON COALESCE(s.teacher_id, o.teacher_id) = u.user_id
                WHERE s.section_id = ?";
        return $this->db->fetchOne($sql, [$id]);
    }
    
    public function getByOffering($offeringId) {
        $sql = "SELECT s.*, u.name as teacher_name
                FROM sections s
                LEFT JOIN users u ON s.teacher_id = u.user_id
                WHERE s.offering_id = ?
                ORDER BY s.section_name ASC";
        return $this->db->fetchAll($sql, [$offeringId]);
    }
    
    public function getByTeacher($teacherId) {
        $sql = "SELECT s.*, o.semester, o.class_time, o.schedule,
                       c.course_name, c.level_code
                FROM sections s
                LEFT JOIN offerings o ON s.offering_id = o.offering_id
                LEFT JOIN courses c ON o.course_id = c.course_id
                WHERE (s.teacher_id = ? OR o.teacher_id = ?) AND s.section_status = 'active'
                ORDER BY o.start_date DESC, s.section_name ASC";
        return $this->db->fetchAll($sql, [$teacherId, $teacherId]);
    }
    
    public function getStudents($sectionId) {
        $sql = "SELECT st.*, e.enrollment_id, e.enrollment_status, e.class_type,
                       e.enrolled_at, e.final_score, e.final_grade, e.attendance_rate
                FROM enrollments e
                JOIN students st ON e.student_id = st.student_id
                WHERE e.section_id = ? AND e.enrollment_status = 'active'
                ORDER BY st.student_no ASC";
        return $this->db->fetchAll($sql, [$sectionId]);
    }
    
    public function getAllStudents($sectionId) {
        $sql = "SELECT st.*, e.enrollment_id, e.enrollment_status, e.class_type,
                       e.enrolled_at, e.final_score, e.final_grade, e.attendance_rate
                FROM enrollments e
                JOIN students st ON e.student_id = st.student_id
                WHERE e.section_id = ?
                ORDER BY e.enrollment_status ASC, st.student_no ASC";
        return $this->db->fetchAll($sql, [$sectionId]);
    }
    
    public function updateCount($sectionId) {
        $sql = "UPDATE sections SET current_count = (
                    SELECT COUNT(*) FROM enrollments 
                    WHERE section_id = ? AND enrollment_status = 'active'
                ) WHERE section_id = ?";
        $this->db->query($sql, [$sectionId, $sectionId]);
    }
    
    public function hasAvailableSlot($sectionId) {
        $section = $this->find($sectionId);
        return $section && $section['current_count'] < $section['max_students'];
    }
    
    public function autoAssign($offeringId, $maxPerSection = 12) {
        $db = $this->db;
        
        // 取得開班資訊
        $offeringModel = new OfferingModel();
        $offering = $offeringModel->getWithDetails($offeringId);
        if (!$offering) {
            throw new Exception('開班資料不存在');
        }
        
        // 取得該程度的學生
        $studentModel = new StudentModel();
        $students = $studentModel->getByLevel($offering['level_code']);
        
        if (empty($students)) {
            return ['sections' => 0, 'students' => 0, 'message' => '沒有符合程度的學生'];
        }
        
        // 過濾已經選過此開班的學生
        $enrolledIds = $this->getEnrolledStudentIds($offeringId);
        $students = array_filter($students, function($s) use ($enrolledIds) {
            return !in_array($s['student_id'], $enrolledIds);
        });
        $students = array_values($students);
        
        if (empty($students)) {
            return ['sections' => 0, 'students' => 0, 'message' => '所有符合程度的學生都已選課'];
        }
        
        // 計算需要的班級數
        $totalStudents = count($students);
        $sectionsNeeded = ceil($totalStudents / $maxPerSection);
        
        $db->beginTransaction();
        try {
            // 取得現有班級
            $existingSections = $this->getByOffering($offeringId);
            $existingCount = count($existingSections);
            
            // 建立新班級
            $sectionLetters = range('A', 'Z');
            for ($i = $existingCount; $i < $sectionsNeeded; $i++) {
                $sectionName = $sectionLetters[$i] . '班';
                $this->create([
                    'offering_id' => $offeringId,
                    'section_name' => $sectionName,
                    'teacher_id' => $offering['teacher_id'],
                    'classroom' => $offering['classroom'],
                    'max_students' => $maxPerSection,
                    'current_count' => 0,
                    'section_status' => 'active'
                ]);
            }
            
            // 重新取得所有班級
            $sections = $this->getByOffering($offeringId);
            
            // 分配學生
            $enrollmentModel = new EnrollmentModel();
            $studentIndex = 0;
            
            foreach ($sections as $section) {
                $availableSlots = $section['max_students'] - $section['current_count'];
                
                for ($i = 0; $i < $availableSlots && $studentIndex < $totalStudents; $i++) {
                    $student = $students[$studentIndex];
                    
                    $enrollmentModel->create([
                        'student_id' => $student['student_id'],
                        'section_id' => $section['section_id'],
                        'enrollment_status' => 'active',
                        'class_type' => 'group',
                        'enrolled_at' => date('Y-m-d H:i:s')
                    ]);
                    
                    $studentIndex++;
                }
                
                // 更新班級人數
                $this->updateCount($section['section_id']);
            }
            
            $db->commit();
            
            return [
                'sections' => count($sections),
                'students' => $studentIndex,
                'message' => "成功分配 {$studentIndex} 位學生到 " . count($sections) . " 個班級"
            ];
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    private function getEnrolledStudentIds($offeringId) {
        $sql = "SELECT DISTINCT e.student_id
                FROM enrollments e
                JOIN sections s ON e.section_id = s.section_id
                WHERE s.offering_id = ?";
        $results = $this->db->fetchAll($sql, [$offeringId]);
        return array_column($results, 'student_id');
    }
}
