<?php
/**
 * Helper - 輔助函數
 */

/**
 * 安全輸出 HTML
 */
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * 產生 URL
 */
function url($path = '') {
    return rtrim(APP_URL, '/') . '/' . ltrim($path, '/');
}

/**
 * 產生資源 URL
 */
function asset($path) {
    return url('public/' . ltrim($path, '/'));
}

/**
 * 格式化日期
 */
function formatDate($date, $format = 'Y-m-d') {
    if (empty($date)) return '';
    return date($format, strtotime($date));
}

/**
 * 格式化日期時間
 */
function formatDateTime($datetime, $format = 'Y-m-d H:i') {
    if (empty($datetime)) return '';
    return date($format, strtotime($datetime));
}

/**
 * 取得 Flash 訊息
 */
function flash($type) {
    $message = $_SESSION['flash'][$type] ?? null;
    unset($_SESSION['flash'][$type]);
    return $message;
}

/**
 * 設定 Flash 訊息
 */
function setFlash($type, $message) {
    $_SESSION['flash'][$type] = $message;
}

/**
 * 判斷是否為目前路徑
 */
function isCurrentPage($page) {
    $current = basename($_SERVER['PHP_SELF']);
    return $current === $page;
}

/**
 * 取得當前使用者
 */
function currentUser() {
    return Auth::getInstance()->user();
}

/**
 * CSRF Token
 */
function csrfToken() {
    return Auth::csrfToken();
}

/**
 * CSRF 表單欄位
 */
function csrfField() {
    return Auth::csrfField();
}

/**
 * 中文程度對照
 */
function getLevelName($code) {
    $levels = [
        'BEG' => '入門華語',
        'BAS' => '基礎華語',
        'ELE1' => '初級華語一',
        'ELE2' => '初級華語二',
        'INT1' => '中級華語一',
        'INT2' => '中級華語二',
        'INT3' => '中級華語三',
        'ADV1' => '進階華語一'
    ];
    return $levels[$code] ?? $code;
}

/**
 * 出席狀態對照
 */
function getAttendanceStatus($code) {
    $statuses = [
        'P' => ['name' => '出席', 'class' => 'success'],
        'L' => ['name' => '遲到', 'class' => 'warning'],
        'E' => ['name' => '請假', 'class' => 'info'],
        'A' => ['name' => '缺席', 'class' => 'danger'],
        'O' => ['name' => '公假', 'class' => 'primary'],
        'S' => ['name' => '病假', 'class' => 'secondary']
    ];
    return $statuses[$code] ?? ['name' => $code, 'class' => 'secondary'];
}

/**
 * 角色名稱對照
 */
function getRoleName($role) {
    $roles = [
        'admin' => '系統管理員',
        'staff' => '教務承辦',
        'teacher' => '老師'
    ];
    return $roles[$role] ?? $role;
}

/**
 * 開班狀態對照
 */
function getOfferingStatus($status) {
    $statuses = [
        'draft' => ['name' => '草稿', 'class' => 'secondary'],
        'active' => ['name' => '進行中', 'class' => 'success'],
        'closed' => ['name' => '已結束', 'class' => 'dark']
    ];
    return $statuses[$status] ?? ['name' => $status, 'class' => 'secondary'];
}

/**
 * 選課狀態對照
 */
function getEnrollmentStatus($status) {
    $statuses = [
        'active' => ['name' => '在學', 'class' => 'success'],
        'dropped' => ['name' => '退選', 'class' => 'warning'],
        'withdrawn' => ['name' => '停修', 'class' => 'danger']
    ];
    return $statuses[$status] ?? ['name' => $status, 'class' => 'secondary'];
}

/**
 * 成績等第對照
 */
function getGradeLetter($score) {
    if ($score === null) return '-';
    if ($score >= 80) return 'A';
    if ($score >= 70) return 'B';
    if ($score >= 60) return 'C';
    return 'D';
}

/**
 * 計算出席率
 */
function calculateAttendanceRate($present, $late, $total) {
    if ($total == 0) return 0;
    return round(($present + $late) / $total * 100, 1);
}

/**
 * 驗證 Email 格式
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * 產生隨機字串
 */
function randomString($length = 32) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * 檔案大小格式化
 */
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = 0;
    while ($bytes >= 1024 && $i < count($units) - 1) {
        $bytes /= 1024;
        $i++;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}

/**
 * 產生分頁 HTML
 */
function renderPagination($pagination, $baseUrl) {
    if ($pagination['total_pages'] <= 1) return '';
    
    $html = '<nav><ul class="pagination justify-content-center mb-0">';
    
    // 上一頁
    if ($pagination['has_prev']) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '&page=' . ($pagination['current_page'] - 1) . '">&laquo;</a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link">&laquo;</span></li>';
    }
    
    // 頁碼
    $start = max(1, $pagination['current_page'] - 2);
    $end = min($pagination['total_pages'], $pagination['current_page'] + 2);
    
    if ($start > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '&page=1">1</a></li>';
        if ($start > 2) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
    }
    
    for ($i = $start; $i <= $end; $i++) {
        if ($i == $pagination['current_page']) {
            $html .= '<li class="page-item active"><span class="page-link">' . $i . '</span></li>';
        } else {
            $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '&page=' . $i . '">' . $i . '</a></li>';
        }
    }
    
    if ($end < $pagination['total_pages']) {
        if ($end < $pagination['total_pages'] - 1) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '&page=' . $pagination['total_pages'] . '">' . $pagination['total_pages'] . '</a></li>';
    }
    
    // 下一頁
    if ($pagination['has_next']) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '&page=' . ($pagination['current_page'] + 1) . '">&raquo;</a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link">&raquo;</span></li>';
    }
    
    $html .= '</ul></nav>';
    
    return $html;
}

/**
 * 輸出 JSON 並結束
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * 記錄操作日誌
 */
function logAudit($action, $tableName = null, $recordId = null, $oldData = null, $newData = null) {
    $db = Database::getInstance();
    $user = currentUser();
    
    $data = [
        'user_id' => $user['user_id'] ?? null,
        'user_account' => $user['account'] ?? null,
        'action' => $action,
        'table_name' => $tableName,
        'record_id' => $recordId,
        'old_data' => $oldData ? json_encode($oldData, JSON_UNESCAPED_UNICODE) : null,
        'new_data' => $newData ? json_encode($newData, JSON_UNESCAPED_UNICODE) : null,
        'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null,
        'create_date' => date('Y-m-d H:i:s')
    ];
    
    $db->insert('audit_logs', $data);
}
