<?php
/**
 * Controller - 基礎控制器類別
 */
class Controller {
    protected $db;
    protected $auth;
    protected $data = [];
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = Auth::getInstance();
        $this->data['auth'] = $this->auth;
        $this->data['currentUser'] = $this->auth->user();
    }
    
    protected function view($view, $data = []) {
        $data = array_merge($this->data, $data);
        extract($data);
        
        $viewFile = VIEW_PATH . '/' . $view . '.php';
        if (!file_exists($viewFile)) {
            throw new Exception("View not found: {$view}");
        }
        
        ob_start();
        include $viewFile;
        $content = ob_get_clean();
        
        // 如果有 layout，包裝內容
        if (isset($data['layout']) && $data['layout'] !== false) {
            $layoutFile = VIEW_PATH . '/layouts/' . ($data['layout'] ?? 'main') . '.php';
            if (file_exists($layoutFile)) {
                include $layoutFile;
            } else {
                echo $content;
            }
        } else {
            echo $content;
        }
    }
    
    protected function json($data, $statusCode = 200) {
        http_response_code($statusCode);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($data, JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    protected function jsonSuccess($data = null, $message = '操作成功') {
        $this->json(['success' => true, 'message' => $message, 'data' => $data]);
    }
    
    protected function jsonError($message = '操作失敗', $errors = [], $statusCode = 400) {
        $this->json(['success' => false, 'message' => $message, 'errors' => $errors], $statusCode);
    }
    
    protected function redirect($url) {
        header("Location: {$url}");
        exit;
    }
    
    protected function requireLogin() {
        if (!$this->auth->check()) {
            if ($this->isAjax()) {
                $this->jsonError('請先登入', [], 401);
            }
            $this->redirect(APP_URL . '/login.php');
        }
    }
    
    protected function requireRole($roles) {
        $this->requireLogin();
        if (!is_array($roles)) {
            $roles = [$roles];
        }
        if (!in_array($this->auth->user()['role'], $roles)) {
            if ($this->isAjax()) {
                $this->jsonError('權限不足', [], 403);
            }
            $this->setFlash('error', '您沒有權限執行此操作');
            $this->redirect(APP_URL . '/dashboard.php');
        }
    }
    
    protected function isAjax() {
        return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
               strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
    }
    
    protected function isPost() {
        return $_SERVER['REQUEST_METHOD'] === 'POST';
    }
    
    protected function input($key, $default = null) {
        return $_POST[$key] ?? $_GET[$key] ?? $default;
    }
    
    protected function inputAll() {
        return array_merge($_GET, $_POST);
    }
    
    protected function validateCsrf() {
        $token = $this->input(CSRF_TOKEN_NAME);
        if (!$token || !isset($_SESSION[CSRF_TOKEN_NAME]) || $token !== $_SESSION[CSRF_TOKEN_NAME]) {
            if ($this->isAjax()) {
                $this->jsonError('安全驗證失敗，請重新整理頁面', [], 403);
            }
            $this->setFlash('error', '安全驗證失敗，請重新操作');
            $this->redirect($_SERVER['HTTP_REFERER'] ?? APP_URL);
        }
    }
    
    protected function setFlash($type, $message) {
        $_SESSION['flash'][$type] = $message;
    }
    
    protected function getFlash($type) {
        $message = $_SESSION['flash'][$type] ?? null;
        unset($_SESSION['flash'][$type]);
        return $message;
    }
    
    protected function paginate($total, $page, $perPage = ITEMS_PER_PAGE) {
        $totalPages = ceil($total / $perPage);
        $page = max(1, min($page, $totalPages));
        $offset = ($page - 1) * $perPage;
        
        return [
            'total' => $total,
            'per_page' => $perPage,
            'current_page' => $page,
            'total_pages' => $totalPages,
            'offset' => $offset,
            'has_prev' => $page > 1,
            'has_next' => $page < $totalPages
        ];
    }
}
