<?php
/**
 * Auth - 認證與授權類別
 */
class Auth {
    private static $instance = null;
    private $user = null;
    private $db;
    
    private function __construct() {
        $this->db = Database::getInstance();
        $this->initSession();
        $this->loadUser();
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function initSession() {
        if (session_status() === PHP_SESSION_NONE) {
            session_name(SESSION_NAME);
            session_set_cookie_params([
                'lifetime' => SESSION_LIFETIME,
                'path' => '/',
                'httponly' => true,
                'samesite' => 'Lax'
            ]);
            session_start();
        }
        
        // 產生 CSRF Token
        if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
            $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
        }
    }
    
    private function loadUser() {
        if (isset($_SESSION['user_id'])) {
            $sql = "SELECT * FROM users WHERE user_id = ? AND status = 1";
            $this->user = $this->db->fetchOne($sql, [$_SESSION['user_id']]);
            
            if (!$this->user) {
                $this->logout();
            }
        }
    }
    
    public function attempt($account, $password) {
        $sql = "SELECT * FROM users WHERE account = ? AND status = 1";
        $user = $this->db->fetchOne($sql, [$account]);
        
        if ($user && password_verify($password, $user['password_hash'])) {
            $this->login($user);
            return true;
        }
        
        return false;
    }
    
    public function login($user) {
        // 重新產生 session id 防止 session fixation
        session_regenerate_id(true);
        
        $_SESSION['user_id'] = $user['user_id'];
        $_SESSION['login_time'] = time();
        
        $this->user = $user;
        
        // 更新最後登入時間
        $this->db->update('users', 
            ['last_login_at' => date('Y-m-d H:i:s')], 
            'user_id = ?', 
            [$user['user_id']]
        );
    }
    
    public function logout() {
        $this->user = null;
        $_SESSION = [];
        
        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', time() - 3600, '/');
        }
        
        session_destroy();
    }
    
    public function check() {
        return $this->user !== null;
    }
    
    public function user() {
        return $this->user;
    }
    
    public function id() {
        return $this->user['user_id'] ?? null;
    }
    
    public function role() {
        return $this->user['role'] ?? null;
    }
    
    public function isAdmin() {
        return $this->role() === 'admin';
    }
    
    public function isStaff() {
        return in_array($this->role(), ['admin', 'staff']);
    }
    
    public function isTeacher() {
        return $this->role() === 'teacher';
    }
    
    public function hasRole($roles) {
        if (!is_array($roles)) {
            $roles = [$roles];
        }
        return in_array($this->role(), $roles);
    }
    
    public function can($permission) {
        $permissions = $this->getPermissions();
        return in_array($permission, $permissions);
    }
    
    private function getPermissions() {
        $role = $this->role();
        
        $permissions = [
            'admin' => [
                'users.manage', 'settings.manage',
                'offerings.create', 'offerings.edit', 'offerings.delete', 'offerings.view',
                'sections.manage', 'students.manage', 'students.import',
                'attendance.manage', 'attendance.lock', 'attendance.unlock',
                'grades.manage', 'transcripts.manage',
                'reports.view', 'emails.manage'
            ],
            'staff' => [
                'offerings.create', 'offerings.edit', 'offerings.view',
                'sections.manage', 'students.manage', 'students.import',
                'attendance.manage', 'attendance.lock', 'attendance.unlock',
                'grades.manage', 'transcripts.manage',
                'reports.view', 'emails.manage'
            ],
            'teacher' => [
                'offerings.view',
                'sections.view', 'students.view',
                'attendance.edit',
                'grades.edit',
                'reports.view'
            ]
        ];
        
        return $permissions[$role] ?? [];
    }
    
    public static function csrfToken() {
        return $_SESSION[CSRF_TOKEN_NAME] ?? '';
    }
    
    public static function csrfField() {
        $token = self::csrfToken();
        return '<input type="hidden" name="' . CSRF_TOKEN_NAME . '" value="' . htmlspecialchars($token) . '">';
    }
}
