<?php
/**
 * OfferingModel - 開班模型
 */
class OfferingModel extends Model {
    protected $table = 'offerings';
    protected $primaryKey = 'offering_id';
    protected $fillable = [
        'course_id', 'semester', 'teacher_id', 'classroom',
        'start_date', 'end_date', 'class_time', 'schedule',
        'hours_per_week', 'total_hours', 'max_sections',
        'max_students_per_section', 'template_id', 'offering_status', 'note', 'updated_by'
    ];
    
    public function getWithDetails($id) {
        $sql = "SELECT o.*, c.course_code, c.course_name, c.course_name_en, c.level_code,
                       u.name as teacher_name, u.user_id as teacher_user_id
                FROM offerings o
                LEFT JOIN courses c ON o.course_id = c.course_id
                LEFT JOIN users u ON o.teacher_id = u.user_id
                WHERE o.offering_id = ?";
        return $this->db->fetchOne($sql, [$id]);
    }
    
    public function getAllWithDetails($semester = '', $status = '', $teacherId = null) {
        $conditions = "1=1";
        $params = [];
        
        if ($semester) {
            $conditions .= " AND o.semester = ?";
            $params[] = $semester;
        }
        
        if ($status) {
            $conditions .= " AND o.offering_status = ?";
            $params[] = $status;
        }
        
        if ($teacherId) {
            $conditions .= " AND o.teacher_id = ?";
            $params[] = $teacherId;
        }
        
        $sql = "SELECT o.*, c.course_code, c.course_name, c.course_name_en, c.level_code,
                       u.name as teacher_name,
                       (SELECT COUNT(*) FROM sections s WHERE s.offering_id = o.offering_id) as section_count,
                       (SELECT COALESCE(SUM(s.current_count), 0) FROM sections s WHERE s.offering_id = o.offering_id) as student_count
                FROM offerings o
                LEFT JOIN courses c ON o.course_id = c.course_id
                LEFT JOIN users u ON o.teacher_id = u.user_id
                WHERE {$conditions}
                ORDER BY o.start_date DESC, c.course_name ASC";
        return $this->db->fetchAll($sql, $params);
    }
    
    public function getSemesters() {
        $sql = "SELECT DISTINCT semester FROM offerings ORDER BY semester DESC";
        return $this->db->fetchAll($sql);
    }
    
    public function getByTeacher($teacherId) {
        $sql = "SELECT o.*, c.course_code, c.course_name, c.level_code
                FROM offerings o
                LEFT JOIN courses c ON o.course_id = c.course_id
                WHERE o.teacher_id = ? OR EXISTS (
                    SELECT 1 FROM sections s WHERE s.offering_id = o.offering_id AND s.teacher_id = ?
                )
                ORDER BY o.start_date DESC";
        return $this->db->fetchAll($sql, [$teacherId, $teacherId]);
    }
    
    public function getActiveOfferings() {
        return $this->getAllWithDetails('', 'active');
    }
    
    public function getSectionCount($offeringId) {
        $sql = "SELECT COUNT(*) FROM sections WHERE offering_id = ?";
        return (int) $this->db->fetchColumn($sql, [$offeringId]);
    }
    
    public function getStudentCount($offeringId) {
        $sql = "SELECT COALESCE(SUM(current_count), 0) FROM sections WHERE offering_id = ?";
        return (int) $this->db->fetchColumn($sql, [$offeringId]);
    }
}
