<?php
/**
 * GradeModel - 成績模型
 */
class GradeModel extends Model {
    protected $table = 'grades';
    protected $primaryKey = 'grade_id';
    protected $fillable = [
        'enrollment_id', 'item_id', 'score', 'weighted_score', 'note',
        'scored_by', 'scored_at', 'updated_by'
    ];
    
    public function getByEnrollment($enrollmentId) {
        $sql = "SELECT g.*, gi.item_name, gi.item_name_en, gi.weight, gi.max_score, gi.sort_order
                FROM grades g
                JOIN grade_items gi ON g.item_id = gi.item_id
                WHERE g.enrollment_id = ?
                ORDER BY gi.sort_order ASC";
        return $this->db->fetchAll($sql, [$enrollmentId]);
    }
    
    public function getGradeItems($offeringId = null) {
        $sql = "SELECT * FROM grade_items 
                WHERE (offering_id = ? OR (offering_id IS NULL AND is_default = 1)) 
                AND status = 1
                ORDER BY sort_order ASC";
        return $this->db->fetchAll($sql, [$offeringId]);
    }
    
    public function getSectionGrades($sectionId) {
        // 取得開班 ID
        $sql = "SELECT offering_id FROM sections WHERE section_id = ?";
        $section = $this->db->fetchOne($sql, [$sectionId]);
        
        if (!$section) return [];
        
        // 取得評分項目
        $items = $this->getGradeItems($section['offering_id']);
        
        // 取得學生與成績
        $sql = "SELECT e.enrollment_id, e.final_score, e.final_grade,
                       st.student_no, st.chinese_name, st.name
                FROM enrollments e
                JOIN students st ON e.student_id = st.student_id
                WHERE e.section_id = ? AND e.enrollment_status = 'active'
                ORDER BY st.student_no ASC";
        $students = $this->db->fetchAll($sql, [$sectionId]);
        
        // 取得所有成績
        $sql = "SELECT g.* 
                FROM grades g
                JOIN enrollments e ON g.enrollment_id = e.enrollment_id
                WHERE e.section_id = ?";
        $grades = $this->db->fetchAll($sql, [$sectionId]);
        
        // 整理成績資料
        $gradeMap = [];
        foreach ($grades as $grade) {
            $gradeMap[$grade['enrollment_id']][$grade['item_id']] = $grade;
        }
        
        foreach ($students as &$student) {
            $student['grades'] = $gradeMap[$student['enrollment_id']] ?? [];
        }
        
        return [
            'items' => $items,
            'students' => $students
        ];
    }
    
    public function saveGrades($sectionId, $gradesData, $scoredBy) {
        $db = $this->db;
        $db->beginTransaction();
        
        try {
            foreach ($gradesData as $enrollmentId => $items) {
                foreach ($items as $itemId => $score) {
                    $score = is_numeric($score) ? (float)$score : null;
                    
                    $existing = $this->whereOne(
                        "enrollment_id = ? AND item_id = ?",
                        [$enrollmentId, $itemId]
                    );
                    
                    // 取得權重計算加權分數
                    $item = $this->db->fetchOne(
                        "SELECT weight FROM grade_items WHERE item_id = ?",
                        [$itemId]
                    );
                    $weightedScore = $score !== null && $item ? $score * $item['weight'] / 100 : null;
                    
                    $data = [
                        'score' => $score,
                        'weighted_score' => $weightedScore,
                        'scored_by' => $scoredBy,
                        'scored_at' => date('Y-m-d H:i:s'),
                        'update_date' => date('Y-m-d H:i:s')
                    ];
                    
                    if ($existing) {
                        $this->update($existing['grade_id'], $data);
                    } else {
                        $data['enrollment_id'] = $enrollmentId;
                        $data['item_id'] = $itemId;
                        $this->create($data);
                    }
                }
                
                // 重新計算最終成績
                $enrollmentModel = new EnrollmentModel();
                $enrollmentModel->calculateFinalScore($enrollmentId);
            }
            
            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    public function createGradeItems($offeringId, $items) {
        $db = $this->db;
        $db->beginTransaction();
        
        try {
            // 刪除現有項目
            $db->delete('grade_items', "offering_id = ?", [$offeringId]);
            
            foreach ($items as $order => $item) {
                $db->insert('grade_items', [
                    'offering_id' => $offeringId,
                    'item_name' => $item['name'],
                    'item_name_en' => $item['name_en'] ?? '',
                    'weight' => $item['weight'],
                    'max_score' => $item['max_score'] ?? 100,
                    'sort_order' => $order + 1,
                    'is_default' => 0,
                    'status' => 1,
                    'create_date' => date('Y-m-d H:i:s')
                ]);
            }
            
            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    public function getGradeDistribution($sectionId) {
        $sql = "SELECT 
                    CASE 
                        WHEN e.final_score >= 80 THEN 'A'
                        WHEN e.final_score >= 70 THEN 'B'
                        WHEN e.final_score >= 60 THEN 'C'
                        WHEN e.final_score IS NOT NULL THEN 'D'
                        ELSE '未評'
                    END as grade,
                    COUNT(*) as count
                FROM enrollments e
                WHERE e.section_id = ? AND e.enrollment_status = 'active'
                GROUP BY grade
                ORDER BY grade";
        return $this->db->fetchAll($sql, [$sectionId]);
    }
    
    public function getGradeReport($filters = []) {
        $conditions = "1=1";
        $params = [];
        
        if (!empty($filters['offering_id'])) {
            $conditions .= " AND o.offering_id = ?";
            $params[] = $filters['offering_id'];
        }
        
        if (!empty($filters['section_id'])) {
            $conditions .= " AND s.section_id = ?";
            $params[] = $filters['section_id'];
        }
        
        $sql = "SELECT 
                    st.student_no, st.chinese_name, st.name as student_name,
                    s.section_name, c.course_name, c.level_code,
                    e.final_score, e.final_grade, e.attendance_rate,
                    e.class_type, e.enrollment_status
                FROM enrollments e
                JOIN students st ON e.student_id = st.student_id
                JOIN sections s ON e.section_id = s.section_id
                JOIN offerings o ON s.offering_id = o.offering_id
                JOIN courses c ON o.course_id = c.course_id
                WHERE {$conditions} AND e.enrollment_status = 'active'
                ORDER BY c.course_name, s.section_name, st.student_no";
        
        return $this->db->fetchAll($sql, $params);
    }
}
