<?php
/**
 * EnrollmentModel - 選課模型
 */
class EnrollmentModel extends Model {
    protected $table = 'enrollments';
    protected $primaryKey = 'enrollment_id';
    protected $fillable = [
        'student_id', 'section_id', 'enrollment_status', 'class_type',
        'enrolled_at', 'dropped_at', 'final_score', 'final_grade',
        'attendance_rate', 'note', 'updated_by'
    ];
    
    public function getWithDetails($id) {
        $sql = "SELECT e.*, st.student_no, st.chinese_name, st.name, st.email,
                       s.section_name, s.offering_id,
                       o.semester, o.class_time, o.schedule,
                       c.course_name, c.level_code
                FROM enrollments e
                JOIN students st ON e.student_id = st.student_id
                JOIN sections s ON e.section_id = s.section_id
                JOIN offerings o ON s.offering_id = o.offering_id
                JOIN courses c ON o.course_id = c.course_id
                WHERE e.enrollment_id = ?";
        return $this->db->fetchOne($sql, [$id]);
    }
    
    public function findByStudentAndSection($studentId, $sectionId) {
        return $this->whereOne("student_id = ? AND section_id = ?", [$studentId, $sectionId]);
    }
    
    public function transfer($enrollmentId, $toSectionId, $operatorId, $reason = '') {
        $db = $this->db;
        
        $enrollment = $this->find($enrollmentId);
        if (!$enrollment) {
            throw new Exception('選課記錄不存在');
        }
        
        $sectionModel = new SectionModel();
        $toSection = $sectionModel->find($toSectionId);
        if (!$toSection) {
            throw new Exception('目標班級不存在');
        }
        
        if (!$sectionModel->hasAvailableSlot($toSectionId)) {
            throw new Exception('目標班級已滿');
        }
        
        $fromSectionId = $enrollment['section_id'];
        
        $db->beginTransaction();
        try {
            // 更新選課記錄
            $this->update($enrollmentId, [
                'section_id' => $toSectionId,
                'update_date' => date('Y-m-d H:i:s')
            ]);
            
            // 記錄異動
            $this->logChange($enrollmentId, $enrollment['student_id'], 'TRANSFER', 
                $fromSectionId, $toSectionId, $reason, $operatorId);
            
            // 更新班級人數
            $sectionModel->updateCount($fromSectionId);
            $sectionModel->updateCount($toSectionId);
            
            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    public function drop($enrollmentId, $operatorId, $reason = '') {
        $db = $this->db;
        
        $enrollment = $this->find($enrollmentId);
        if (!$enrollment) {
            throw new Exception('選課記錄不存在');
        }
        
        $db->beginTransaction();
        try {
            $this->update($enrollmentId, [
                'enrollment_status' => 'dropped',
                'dropped_at' => date('Y-m-d H:i:s')
            ]);
            
            $this->logChange($enrollmentId, $enrollment['student_id'], 'DROP',
                $enrollment['section_id'], null, $reason, $operatorId);
            
            $sectionModel = new SectionModel();
            $sectionModel->updateCount($enrollment['section_id']);
            
            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    public function withdraw($enrollmentId, $operatorId, $reason = '') {
        $db = $this->db;
        
        $enrollment = $this->find($enrollmentId);
        if (!$enrollment) {
            throw new Exception('選課記錄不存在');
        }
        
        $db->beginTransaction();
        try {
            $this->update($enrollmentId, [
                'enrollment_status' => 'withdrawn',
                'dropped_at' => date('Y-m-d H:i:s')
            ]);
            
            $this->logChange($enrollmentId, $enrollment['student_id'], 'WITHDRAW',
                $enrollment['section_id'], null, $reason, $operatorId);
            
            $sectionModel = new SectionModel();
            $sectionModel->updateCount($enrollment['section_id']);
            
            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    public function addToSection($studentId, $sectionId, $operatorId) {
        $db = $this->db;
        
        $sectionModel = new SectionModel();
        if (!$sectionModel->hasAvailableSlot($sectionId)) {
            throw new Exception('班級已滿');
        }
        
        // 檢查是否已選
        if ($this->findByStudentAndSection($studentId, $sectionId)) {
            throw new Exception('學生已在此班級');
        }
        
        $db->beginTransaction();
        try {
            $enrollmentId = $this->create([
                'student_id' => $studentId,
                'section_id' => $sectionId,
                'enrollment_status' => 'active',
                'class_type' => 'group',
                'enrolled_at' => date('Y-m-d H:i:s')
            ]);
            
            $this->logChange($enrollmentId, $studentId, 'ADD', null, $sectionId, '補選加入', $operatorId);
            
            $sectionModel->updateCount($sectionId);
            
            $db->commit();
            return $enrollmentId;
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    private function logChange($enrollmentId, $studentId, $changeType, $fromSectionId, $toSectionId, $reason, $operatorId) {
        $data = [
            'enrollment_id' => $enrollmentId,
            'student_id' => $studentId,
            'change_type' => $changeType,
            'from_section_id' => $fromSectionId,
            'to_section_id' => $toSectionId,
            'reason' => $reason,
            'operator_id' => $operatorId,
            'notification_status' => 'pending',
            'changed_at' => date('Y-m-d H:i:s'),
            'create_date' => date('Y-m-d H:i:s')
        ];
        $this->db->insert('enrollment_changes', $data);
    }
    
    public function getChangeHistory($enrollmentId) {
        $sql = "SELECT ec.*, 
                       fs.section_name as from_section_name,
                       ts.section_name as to_section_name,
                       u.name as operator_name
                FROM enrollment_changes ec
                LEFT JOIN sections fs ON ec.from_section_id = fs.section_id
                LEFT JOIN sections ts ON ec.to_section_id = ts.section_id
                LEFT JOIN users u ON ec.operator_id = u.user_id
                WHERE ec.enrollment_id = ?
                ORDER BY ec.changed_at DESC";
        return $this->db->fetchAll($sql, [$enrollmentId]);
    }
    
    public function calculateFinalScore($enrollmentId) {
        $sql = "SELECT SUM(g.score * gi.weight / 100) as final_score
                FROM grades g
                JOIN grade_items gi ON g.item_id = gi.item_id
                WHERE g.enrollment_id = ? AND g.score IS NOT NULL";
        $result = $this->db->fetchOne($sql, [$enrollmentId]);
        $finalScore = $result ? round($result['final_score'], 2) : null;
        
        if ($finalScore !== null) {
            $this->update($enrollmentId, [
                'final_score' => $finalScore,
                'final_grade' => getGradeLetter($finalScore)
            ]);
        }
        
        return $finalScore;
    }
    
    public function calculateAttendanceRate($enrollmentId) {
        $enrollment = $this->find($enrollmentId);
        if (!$enrollment) return null;
        
        $sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN attendance_status IN ('P', 'L') THEN 1 ELSE 0 END) as present
                FROM attendance
                WHERE section_id = ? AND student_id = (
                    SELECT student_id FROM enrollments WHERE enrollment_id = ?
                )";
        $result = $this->db->fetchOne($sql, [$enrollment['section_id'], $enrollmentId]);
        
        $rate = $result['total'] > 0 ? round($result['present'] / $result['total'] * 100, 1) : null;
        
        if ($rate !== null) {
            $this->update($enrollmentId, ['attendance_rate' => $rate]);
        }
        
        return $rate;
    }
}
