<?php
/**
 * CourseModel - 課程模型
 */
class CourseModel extends Model {
    protected $table = 'courses';
    protected $primaryKey = 'course_id';
    protected $fillable = [
        'course_code', 'course_name', 'course_name_en', 'level_code',
        'textbook', 'description', 'status', 'updated_by'
    ];
    
    public function getActive() {
        return $this->where("status = 1", [], "level_code ASC, course_name ASC");
    }
    
    public function getByLevel($levelCode) {
        return $this->where("level_code = ? AND status = 1", [$levelCode], "course_name ASC");
    }
    
    public function search($keyword = '', $level = '') {
        $conditions = "1=1";
        $params = [];
        
        if ($keyword) {
            $conditions .= " AND (course_code LIKE ? OR course_name LIKE ? OR course_name_en LIKE ?)";
            $params = array_merge($params, ["%{$keyword}%", "%{$keyword}%", "%{$keyword}%"]);
        }
        
        if ($level) {
            $conditions .= " AND level_code = ?";
            $params[] = $level;
        }
        
        return $this->where($conditions, $params, "level_code ASC, course_name ASC");
    }
}

/**
 * LevelModel - 程度模型
 */
class LevelModel extends Model {
    protected $table = 'levels';
    protected $primaryKey = 'level_id';
    protected $fillable = [
        'level_code', 'level_name', 'level_name_en', 'textbook', 'sort_order', 'status', 'updated_by'
    ];
    
    public function getActive() {
        return $this->where("status = 1", [], "sort_order ASC");
    }
    
    public function getByCode($code) {
        return $this->findBy('level_code', $code);
    }
}

/**
 * SettingModel - 系統設定模型
 */
class SettingModel extends Model {
    protected $table = 'system_settings';
    protected $primaryKey = 'setting_id';
    
    public function get($key, $default = null) {
        $setting = $this->findBy('setting_key', $key);
        if (!$setting) return $default;
        
        $value = $setting['setting_value'];
        switch ($setting['setting_type']) {
            case 'int':
                return (int) $value;
            case 'float':
                return (float) $value;
            case 'boolean':
                return in_array(strtolower($value), ['1', 'true', 'yes']);
            case 'json':
                return json_decode($value, true);
            default:
                return $value;
        }
    }
    
    public function set($key, $value, $type = 'string') {
        if ($type === 'json') {
            $value = json_encode($value, JSON_UNESCAPED_UNICODE);
        } elseif ($type === 'boolean') {
            $value = $value ? '1' : '0';
        }
        
        $existing = $this->findBy('setting_key', $key);
        
        if ($existing) {
            return $this->db->update($this->table, [
                'setting_value' => $value,
                'setting_type' => $type,
                'update_date' => date('Y-m-d H:i:s')
            ], "setting_key = ?", [$key]);
        } else {
            return $this->db->insert($this->table, [
                'setting_key' => $key,
                'setting_value' => $value,
                'setting_type' => $type,
                'create_date' => date('Y-m-d H:i:s')
            ]);
        }
    }
    
    public function getAll() {
        $settings = $this->all('setting_key ASC');
        $result = [];
        foreach ($settings as $setting) {
            $result[$setting['setting_key']] = $this->get($setting['setting_key']);
        }
        return $result;
    }
}
