<?php
/**
 * AttendanceModel - 點名模型
 */
class AttendanceModel extends Model {
    protected $table = 'attendance';
    protected $primaryKey = 'attendance_id';
    protected $fillable = [
        'section_id', 'student_id', 'class_date', 'attendance_status',
        'hours', 'note', 'is_locked', 'locked_at', 'locked_by', 'recorded_by', 'updated_by'
    ];
    
    public function getByDate($sectionId, $classDate) {
        $sql = "SELECT a.*, st.student_no, st.chinese_name, st.name
                FROM attendance a
                JOIN students st ON a.student_id = st.student_id
                WHERE a.section_id = ? AND a.class_date = ?
                ORDER BY st.student_no ASC";
        return $this->db->fetchAll($sql, [$sectionId, $classDate]);
    }
    
    public function getByStudent($studentId, $sectionId = null) {
        $conditions = "a.student_id = ?";
        $params = [$studentId];
        
        if ($sectionId) {
            $conditions .= " AND a.section_id = ?";
            $params[] = $sectionId;
        }
        
        $sql = "SELECT a.*, s.section_name, c.course_name
                FROM attendance a
                JOIN sections s ON a.section_id = s.section_id
                JOIN offerings o ON s.offering_id = o.offering_id
                JOIN courses c ON o.course_id = c.course_id
                WHERE {$conditions}
                ORDER BY a.class_date DESC";
        return $this->db->fetchAll($sql, $params);
    }
    
    public function getDates($sectionId) {
        $sql = "SELECT ad.*, 
                       (SELECT COUNT(*) FROM attendance a WHERE a.section_id = ad.section_id AND a.class_date = ad.class_date) as record_count
                FROM attendance_dates ad
                WHERE ad.section_id = ?
                ORDER BY ad.class_date DESC";
        return $this->db->fetchAll($sql, [$sectionId]);
    }
    
    public function createDate($sectionId, $classDate, $hours = null) {
        // 檢查是否已存在
        $sql = "SELECT date_id FROM attendance_dates WHERE section_id = ? AND class_date = ?";
        $existing = $this->db->fetchOne($sql, [$sectionId, $classDate]);
        
        if ($existing) {
            return $existing['date_id'];
        }
        
        return $this->db->insert('attendance_dates', [
            'section_id' => $sectionId,
            'class_date' => $classDate,
            'hours' => $hours,
            'is_locked' => 0,
            'create_date' => date('Y-m-d H:i:s')
        ]);
    }
    
    public function initializeAttendance($sectionId, $classDate, $recordedBy) {
        // 取得班級學生
        $sectionModel = new SectionModel();
        $students = $sectionModel->getStudents($sectionId);
        
        // 建立點名日期
        $this->createDate($sectionId, $classDate);
        
        $created = 0;
        foreach ($students as $student) {
            // 檢查是否已有點名記錄
            $existing = $this->whereOne(
                "section_id = ? AND student_id = ? AND class_date = ?",
                [$sectionId, $student['student_id'], $classDate]
            );
            
            if (!$existing) {
                $this->create([
                    'section_id' => $sectionId,
                    'student_id' => $student['student_id'],
                    'class_date' => $classDate,
                    'attendance_status' => 'P', // 預設出席
                    'recorded_by' => $recordedBy
                ]);
                $created++;
            }
        }
        
        return $created;
    }
    
    public function saveAttendance($sectionId, $classDate, $attendanceData, $recordedBy) {
        // 檢查是否鎖定
        if ($this->isDateLocked($sectionId, $classDate)) {
            throw new Exception('點名記錄已鎖定，無法修改');
        }
        
        $db = $this->db;
        $db->beginTransaction();
        
        try {
            foreach ($attendanceData as $studentId => $data) {
                $existing = $this->whereOne(
                    "section_id = ? AND student_id = ? AND class_date = ?",
                    [$sectionId, $studentId, $classDate]
                );
                
                $record = [
                    'attendance_status' => $data['status'] ?? 'P',
                    'note' => $data['note'] ?? '',
                    'recorded_by' => $recordedBy,
                    'update_date' => date('Y-m-d H:i:s')
                ];
                
                if ($existing) {
                    $this->update($existing['attendance_id'], $record);
                } else {
                    $record['section_id'] = $sectionId;
                    $record['student_id'] = $studentId;
                    $record['class_date'] = $classDate;
                    $this->create($record);
                }
            }
            
            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
    public function isDateLocked($sectionId, $classDate) {
        $sql = "SELECT is_locked FROM attendance_dates WHERE section_id = ? AND class_date = ?";
        $result = $this->db->fetchOne($sql, [$sectionId, $classDate]);
        return $result && $result['is_locked'] == 1;
    }
    
    public function lockDate($sectionId, $classDate, $userId) {
        return $this->db->update('attendance_dates', [
            'is_locked' => 1,
            'locked_at' => date('Y-m-d H:i:s'),
            'locked_by' => $userId,
            'update_date' => date('Y-m-d H:i:s')
        ], "section_id = ? AND class_date = ?", [$sectionId, $classDate]);
    }
    
    public function unlockDate($sectionId, $classDate) {
        return $this->db->update('attendance_dates', [
            'is_locked' => 0,
            'locked_at' => null,
            'locked_by' => null,
            'update_date' => date('Y-m-d H:i:s')
        ], "section_id = ? AND class_date = ?", [$sectionId, $classDate]);
    }
    
    public function getStats($sectionId, $studentId = null) {
        $conditions = "section_id = ?";
        $params = [$sectionId];
        
        if ($studentId) {
            $conditions .= " AND student_id = ?";
            $params[] = $studentId;
        }
        
        $sql = "SELECT 
                    student_id,
                    COUNT(*) as total,
                    SUM(CASE WHEN attendance_status = 'P' THEN 1 ELSE 0 END) as present,
                    SUM(CASE WHEN attendance_status = 'L' THEN 1 ELSE 0 END) as late,
                    SUM(CASE WHEN attendance_status = 'A' THEN 1 ELSE 0 END) as absent,
                    SUM(CASE WHEN attendance_status = 'E' THEN 1 ELSE 0 END) as excused,
                    SUM(CASE WHEN attendance_status = 'O' THEN 1 ELSE 0 END) as official,
                    SUM(CASE WHEN attendance_status = 'S' THEN 1 ELSE 0 END) as sick
                FROM attendance
                WHERE {$conditions}
                GROUP BY student_id";
        
        if ($studentId) {
            return $this->db->fetchOne($sql, $params);
        }
        
        return $this->db->fetchAll($sql, $params);
    }
    
    public function getSectionStats($sectionId) {
        $stats = $this->getStats($sectionId);
        $result = [];
        
        foreach ($stats as $stat) {
            $total = $stat['total'];
            $attended = $stat['present'] + $stat['late'];
            $rate = $total > 0 ? round($attended / $total * 100, 1) : 0;
            
            $result[$stat['student_id']] = array_merge($stat, [
                'attendance_rate' => $rate
            ]);
        }
        
        return $result;
    }
    
    public function getReport($filters = []) {
        $conditions = "1=1";
        $params = [];
        
        if (!empty($filters['start_date'])) {
            $conditions .= " AND a.class_date >= ?";
            $params[] = $filters['start_date'];
        }
        
        if (!empty($filters['end_date'])) {
            $conditions .= " AND a.class_date <= ?";
            $params[] = $filters['end_date'];
        }
        
        if (!empty($filters['offering_id'])) {
            $conditions .= " AND o.offering_id = ?";
            $params[] = $filters['offering_id'];
        }
        
        if (!empty($filters['section_id'])) {
            $conditions .= " AND s.section_id = ?";
            $params[] = $filters['section_id'];
        }
        
        if (!empty($filters['level_code'])) {
            $conditions .= " AND c.level_code = ?";
            $params[] = $filters['level_code'];
        }
        
        $sql = "SELECT 
                    st.student_no, st.chinese_name, st.name as student_name,
                    s.section_name, c.course_name, c.level_code,
                    COUNT(*) as total_classes,
                    SUM(CASE WHEN a.attendance_status = 'P' THEN 1 ELSE 0 END) as present,
                    SUM(CASE WHEN a.attendance_status = 'L' THEN 1 ELSE 0 END) as late,
                    SUM(CASE WHEN a.attendance_status = 'A' THEN 1 ELSE 0 END) as absent,
                    SUM(CASE WHEN a.attendance_status IN ('E','O','S') THEN 1 ELSE 0 END) as leave_count,
                    ROUND((SUM(CASE WHEN a.attendance_status IN ('P','L') THEN 1 ELSE 0 END) / COUNT(*)) * 100, 1) as attendance_rate
                FROM attendance a
                JOIN students st ON a.student_id = st.student_id
                JOIN sections s ON a.section_id = s.section_id
                JOIN offerings o ON s.offering_id = o.offering_id
                JOIN courses c ON o.course_id = c.course_id
                WHERE {$conditions}
                GROUP BY a.student_id, a.section_id
                ORDER BY c.course_name, s.section_name, st.student_no";
        
        return $this->db->fetchAll($sql, $params);
    }
}
